/// common functions used by most Synopse projects
// - this unit is a part of the freeware Synopse framework,
// licensed under a MPL/GPL/LGPL tri-license; version 1.8
unit SynCommons;

{
    This file is part of Synopse framework.

    Synopse framework. Copyright (C) 2010 Arnaud Bouchez
      Synopse Informatique - http://synopse.info

  *** BEGIN LICENSE BLOCK *****
  Version: MPL 1.1/GPL 2.0/LGPL 2.1

  The contents of this file are subject to the Mozilla Public License Version
  1.1 (the "License"); you may not use this file except in compliance with
  the License. You may obtain a copy of the License at
  http://www.mozilla.org/MPL

  Software distributed under the License is distributed on an "AS IS" basis,
  WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
  for the specific language governing rights and limitations under the License.

  The Original Code is Synopse framework.

  The Initial Developer of the Original Code is Arnaud Bouchez.

  Portions created by the Initial Developer are Copyright (C) 2010
  the Initial Developer. All Rights Reserved.

  Contributor(s):
  Alternatively, the contents of this file may be used under the terms of
  either the GNU General Public License Version 2 or later (the "GPL"), or
  the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
  in which case the provisions of the GPL or the LGPL are applicable instead
  of those above. If you wish to allow use of your version of this file only
  under the terms of either the GPL or the LGPL, and not to allow others to
  use your version of this file under the terms of the MPL, indicate your
  decision by deleting the provisions above and replace them with the notice
  and other provisions required by the GPL or the LGPL. If you do not delete
  the provisions above, a recipient may use your version of this file under
  the terms of any one of the MPL, the GPL or the LGPL.

  ***** END LICENSE BLOCK *****

  Version 1.7
  - first public release, corresponding to SQLite3 Framework 1.7

  Version 1.8
  - includes Unitary Testing class and functions
  - bug fixed in WinAnsiBufferToUtf8() and all WinAnsi to UTF-8 encoding
    functions (issue identified thanks to new _UTF8 testing function)
  - bug fixed in val() under Delphi 2009/2010 for some values (issue identified
    thanks to new NumericalConversion testing function)
  - bug fixed in AnsiICompW() - used in SynPdf unit
  - ambiguous SameText() function rewritten as SameTextU() with UTF-8 decoding
  - TTextWriter class moved from SQLite3Commons to SynCommons
  - new JSONEncode and JSONDecode functions to directly encode or decode any
    content to/from a valid UTF-8 JSON object content
  - enhanced URLEncode() and URLDecode() functions
  - new ExtendedToStr/ExtendedToString functions
  - new tests added (mostly relative to the new functions or classes)

}


{$I Synopse.inc} // define HASINLINE USETYPEINFO CPU32 CPU64 OWNNORMTOUPPER

interface

uses
{$ifdef MSWINDOWS}
  Windows,
  Messages,
{$endif}
{$ifdef LINUX} // for Kylix
  Libc,
  Types,
{$endif}
  Classes,
{$ifndef LVCL}
  Contnrs, // for TObjectList
{$endif}
  SysUtils;

  
const
  /// the corresponding version of the freeware Synopse framework
  SYNOPSE_FRAMEWORK_VERSION = '1.8';


{ ************ common types used for compatibility between compilers and CPU }

{$ifndef FPC}
type
  /// a CPU-dependent unsigned integer type cast of a pointer / register
  // - used for 64 bits compatibility, native under Free Pascal Compiler
  PtrUInt = cardinal;
  /// a CPU-dependent unsigned integer type cast of a pointer of pointer
  // - used for 64 bits compatibility, native under Free Pascal Compiler
  PPtrUInt = ^PtrUInt;

  /// a CPU-dependent signed integer type cast of a pointer / register
  // - used for 64 bits compatibility, native under Free Pascal Compiler
  PtrInt = integer;
  /// a CPU-dependent signed integer type cast of a pointer of pointer
  // - used for 64 bits compatibility, native under Free Pascal Compiler
  PPtrInt = ^PtrInt;

  /// unsigned Int64 doesn't exist under Delphi, but is defined in FPC
  QWord = Int64;
{$endif}

type
  {{ RawUnicode is an Unicode String stored in an AnsiString
    - faster than WideString, which are allocated in Global heap (for COM)
    - an AnsiChar(#0) is added at the end, for having a true WideChar(#0) at ending
    - length(RawUnicode) returns memory bytes count: use (length(RawUnicode) shr 1)
     for WideChar count (that's why the definition of this type since Delphi 2009
     is AnsiString(1200) and not UnicodeString)
    - pointer(RawUnicode) is compatible with Win32 'Wide' API call
    - mimic Delphi 2009 UnicodeString, without the WideString or Ansi conversion overhead
    - all conversion to/from AnsiString or RawUTF8 must be explicit }
{$ifdef UNICODE} RawUnicode = type AnsiString(1200); // Codepage for an UnicodeString
{$else}          RawUnicode = type AnsiString;
{$endif}

  {{ RawUTF8 is an UTF-8 String stored in an AnsiString
    - use this type instead of System.UTF8String, which behavior changed
     between Delphi 2009 compiler and previous versions: our implementation
     is consistent and compatible with all versions of Delphi compiler
    - mimic Delphi 2009 UTF8String, without the charset conversion overhead
    - all conversion to/from AnsiString or RawUnicode must be explicit }
{$ifdef UNICODE} RawUTF8 = type AnsiString(CP_UTF8); // Codepage for an UTF8string
{$else}          RawUTF8 = type AnsiString; {$endif}

  {{ WinAnsiString is a WinAnsi-encoded AnsiString (code page 1252)
    - use this type instead of System.String, which behavior changed
     between Delphi 2009 compiler and previous versions: our implementation
     is consistent and compatible with all versions of Delphi compiler
    - all conversion to/from RawUTF8 or RawUnicode must be explicit }
{$ifdef UNICODE} WinAnsiString = type AnsiString(1252); // WinAnsi Codepage
{$else}          WinAnsiString = type AnsiString; {$endif}

{$ifndef UNICODE}
  /// define RawByteString, as it does exist in Delphi 2009/2010
  // - to be used for byte storage into an AnsiString
  // - use this type if you don't want the Delphi compiler not to do any
  // code page conversions when you assign a typed AnsiString to a RawByteString,
  // i.e. a RawUTF8 or a WinAnsiString
  RawByteString = AnsiString;
{$endif}

  {{ SynUnicode is the fastest available Unicode native string type, depending
    on the compiler used
   - this type is native to the compiler, so you can use Length() Copy() and
     such functions with it (this is not possible with RawUnicodeString type)
   - before Delphi 2009/2010, it uses slow OLE compatible WideString
     (with our Enhanced RTL, WideString allocation can be made faster by using
     an internal caching mechanism)
   - starting with Delphi 2009/2010, it uses fastest UnicodeString type, which
     allow Copy On Write, Reference Counting and fast heap memory allocation }
  {$ifdef UNICODE}SynUnicode = UnicodeString;
  {$else}         SynUnicode = WideString; {$endif}

  PRawUnicode = ^RawUnicode;
  PRawUTF8 = ^RawUTF8;
  PWinAnsiString = ^WinAnsiString;
  PWinAnsiChar = type PAnsiChar;

  /// a simple wrapper to UTF-8 encoded zero-terminated PAnsiChar
  // - PAnsiChar is used only for Win-Ansi encoded text
  // - the Synopse SQLite3 framework uses mostly this PUTF8Char type,
  // because all data is internaly stored and expected to be UTF-8 encoded 
  PUTF8Char = type PAnsiChar;
  PPUTF8Char = ^PUTF8Char;

  /// a Row/Col array of PUTF8Char, for containing sqlite3_get_table() result
  TPUtf8CharArray = array[0..MaxInt div SizeOf(PUTF8Char)-1] of PUTF8Char;
  PPUtf8CharArray = ^TPUtf8CharArray;

  /// a dynamic array of PUTF8Char pointers
  TPUtf8CharDynArray = array of PUTF8Char;

  /// a dynamic array of UTF-8 encoded strings
  TRawUTF8DynArray = array of RawUTF8;
  PRawUTF8DynArray = ^TRawUTF8DynArray;

  TWordArray  = array[0..MaxInt div SizeOf(word)-1] of word;
  PWordArray = ^TWordArray;

  TInt64Array = array[0..MaxInt div SizeOf(Int64)-1] of Int64;
  PInt64Array = ^TInt64Array;

  TCardinalArray = array[0..MaxInt div SizeOf(cardinal)-1] of cardinal;
  PCardinalArray = ^TCardinalArray;


{ ************ fast UTF-8 / Unicode / Ansi types and conversion routines }

var
  /// this table contain all the unicode characters corresponding to
  // the Ansi Code page 1252 (i.e. what we call WinAnsi within the framework)
  WinAnsiTable: packed array[byte] of Word;

const
  /// MIME content type used for JSON communication (as used by the Microsoft
  // WCF framework and the YUI framework)
  JSON_CONTENT_TYPE = 'application/json; charset=UTF-8';

  /// MIME content type used for plain UTF-8 text
  TEXT_CONTENT_TYPE = 'text/plain; charset="UTF-8"';

  /// used for fast WinAnsi to Unicode conversion
  // - this table contain all the unicode characters corresponding to
  // the Ansi Code page 1252 (i.e. WinAnsi), which unicode value are > 255
  // - values taken from MultiByteToWideChar(1252,0,@Tmp,256,@WinAnsiTable,256)
  // so these values are available outside the Windows platforms (e.g. Linux/BSD)
  WinAnsiUnicodeChars: packed array[128..159] of cardinal =
    (8364, 129, 8218, 402, 8222, 8230, 8224, 8225, 710, 8240, 352, 8249, 338,
     141, 381, 143, 144, 8216, 8217, 8220, 8221, 8226, 8211, 8212, 732, 8482,
     353, 8250, 339, 157, 382, 376);

  /// used for fast Unicode to WinAnsi conversion
  //-  this table contain all the unicode values corresponding to
  // the Ansi Code page 1252 (i.e. WinAnsi), which unicode value are > 255,
  // sorted by increasing order (you can use a fast binary search to lookup
  // for the corresponding Ansi char, with WinAnsiTableSortedAnsi[] below)
  WinAnsiTableSortedWide: array[0..26] of integer =
    (338, 339, 352, 353, 376, 381, 382, 402, 710,
     732, 8211, 8212, 8216, 8217, 8218, 8220, 8221, 8222, 8224, 8225, 8226,
     8230, 8240, 8249, 8250, 8364, 8482);

  /// used for fast Unicode to WinAnsi conversion
  // - lookup table for every WinAnsi char corresponding to a WinAnsiTableSortedWide[]
  WinAnsiTableSortedAnsi: array[0..26] of byte =
    (140, 156, 138, 154, 159, 142, 158, 131, 136,
     152, 150, 151, 145, 146, 130, 147, 148, 132, 134, 135, 149, 133, 137,
     139, 155, 128, 153);


/// conversion of a wide char into a WinAnsi (CodePage 1252) char
// - return ' ' for an unknown WideChar in code page 1252
function WideCharToWinAnsiChar(wc: cardinal): AnsiChar; {$ifdef HASINLINE}inline;{$endif}

/// conversion of a wide char into a WinAnsi (CodePage 1252) char index
// - return -1 for an unknown WideChar in code page 1252
function WideCharToWinAnsi(wc: cardinal): integer; {$ifdef HASINLINE}inline;{$endif}

/// internal function called by WideCharToWinAnsi()
// - should not be used in your code, only here to allow inlining of the
// WideCharToWinAnsi function
function WinAnsiTableSortedFind(wc: cardinal): Integer;

/// return TRUE if the supplied buffer only contains 7-bits Ansi characters
function IsAnsiCompatible(PC: PAnsiChar): boolean;

/// return TRUE if the supplied unicode buffer only contains WinAnsi characters
// - i.e. if the text can be displayed using ANSI_CHARSET
function IsWinAnsi(WideText: PWideChar): boolean; overload;

/// return TRUE if the supplied unicode buffer only contains WinAnsi characters
// - i.e. if the text can be displayed using ANSI_CHARSET
function IsWinAnsi(WideText: PWideChar; Length: integer): boolean; overload;

/// return TRUE if the supplied UTF-8 buffer only contains WinAnsi characters
// - i.e. if the text can be displayed using ANSI_CHARSET
function IsWinAnsiU(UTF8Text: PUTF8Char): boolean; 

/// UTF-8 encode one Unicode character into Dest
// - return the number of bytes written into Dest (i.e. 1,2 or 3) 
function UnicodeCharToUtf8(Dest: PUTF8Char; aWideChar: PtrUInt): integer;
  {$ifdef HASINLINE}inline;{$endif}

/// direct conversion of a WinAnsi (CodePage 1252) string into a UTF-8 encoded String
// - faster than SysUtils: don't use Utf8Encode(WideString) -> no Windows.Global(),
// and use a fixed pre-calculated array for individual chars conversion
function WinAnsiToUtf8(const S: WinAnsiString): RawUTF8;

/// direct conversion of a WinAnsi PAnsiChar buffer into a UTF-8 encoded buffer
// - Dest^ buffer must be reserved with at least SourceChars*3
function WinAnsiBufferToUtf8(Dest: PUTF8Char; Source: PAnsiChar; SourceChars: Cardinal): PUTF8Char;

/// direct conversion of a WinAnsi (CodePage 1252) string into a Unicode encoded String
// - very fast, by using a fixed pre-calculated array for individual chars conversion
function WinAnsiToRawUnicode(const S: WinAnsiString): RawUnicode;

/// direct conversion of a WinAnsi (CodePage 1252) string into a Unicode buffer
// - very fast, by using a fixed pre-calculated array for individual chars conversion
procedure WinAnsiToUnicodeBuffer(const S: WinAnsiString; Dest: PWordArray; DestLen: integer);

/// direct conversion of a UTF-8 encoded string into a WinAnsi String
function Utf8ToWinAnsi(const S: RawUTF8): WinAnsiString; overload;

/// direct conversion of a UTF-8 encoded zero terminated buffer into a WinAnsi String
function Utf8ToWinAnsi(P: PUTF8Char): WinAnsiString; overload;

/// direct conversion of a UTF-8 encoded zero terminated buffer into a RawUTF8 String
procedure Utf8ToRawUTF8(P: PUTF8Char; var result: RawUTF8);
  {$ifdef HASINLINE}inline;{$endif}

/// direct conversion of a UTF-8 encoded buffer into a WinAnsi PAnsiChar buffer
function UTF8ToWinPChar(dest: PAnsiChar; source: PUTF8Char; count: integer): integer;

/// direct conversion of a UTF-8 encoded buffer into a WinAnsi shortstring buffer
procedure UTF8ToShortString(var dest: shortstring; source: PUTF8Char);

/// convert an UTF-8 encoded text into a WideChar array
// - faster than System.UTF8ToUnicode
// - sourceBytes can by 0, therefore length is computed from zero terminated source
// - enough place must be available in dest
// - a WideChar(#0) is added at the end (if something is written)
// - returns the byte count written in dest, excluding the ending WideChar(#0)
function UTF8ToWideChar(dest: pWideChar; source: PUTF8Char; sourceBytes: PtrInt=0): PtrInt;

/// calculate the character count UTF-8 encoded in source^
// - faster than System.UTF8ToUnicode with dest=nil
function Utf8ToUnicodeLength(source: PUTF8Char): PtrInt;

/// calculate the character count of the first line UTF-8 encoded in source^
// - end the count at first #13 or #10 character
function Utf8FirstLineToUnicodeLength(source: PUTF8Char): PtrInt;

/// convert a UTF-8 encoded buffer into a RawUnicode string
// - if L is 0, L is computed from zero terminated P buffer
// - RawUnicode is ended by a WideChar(#0)
// - faster than System.Utf8Decode() which uses slow widestrings
function Utf8DecodeToRawUnicode(P: PUTF8Char; L: integer): RawUnicode; overload;

/// convert a UTF-8 string into a RawUnicode string
function Utf8DecodeToRawUnicode(const S: RawUTF8): RawUnicode; overload;

/// convert a UTF-8 string into a RawUnicode string
// - this version doesn't resize the length of the result RawUnicode
// and is therefore usefull before a Win32 Unicode API call (with nCount=-1)
function Utf8DecodeToRawUnicodeUI(const S: RawUTF8): RawUnicode; overload;

/// convert a RawUnicode PWideChar into a UTF-8 string
function RawUnicodeToUtf8(P: PWideChar; WideCharCount: integer): RawUTF8; overload;

/// convert a RawUnicode PWideChar into a UTF-8 buffer
// - replace system.UnicodeToUtf8 implementation, which is rather slow
// since Delphi 2009/2010
function RawUnicodeToUtf8(Dest: PUTF8Char; DestLen: PtrInt; Source: PWideChar; SourceLen: PtrInt): PtrInt; overload;

/// convert a RawUnicode PWideChar into a UTF-8 string
// - this version doesn't resize the resulting RawUTF8 string, but return
// the new resulting RawUTF8 byte count into UTF8Length
function RawUnicodeToUtf8(P: PWideChar; WideCharCount: integer; out UTF8Length: integer): RawUTF8; overload;

/// convert a RawUnicode string into a UTF-8 string
function RawUnicodeToUtf8(const Unicode: RawUnicode): RawUTF8; overload;

/// direct conversion of a Unicode encoded buffer into a WinAnsi PAnsiChar buffer
procedure RawUnicodeToWinPChar(dest: PAnsiChar; source: PWideChar; WideCharCount: integer);

/// convert a RawUnicode PWideChar into a WinAnsi (code page 1252) string
function RawUnicodeToWinAnsi(P: PWideChar; WideCharCount: integer): WinAnsiString; overload;

/// convert a RawUnicode string into a WinAnsi (code page 1252) string
function RawUnicodeToWinAnsi(const Unicode: RawUnicode): WinAnsiString; overload;

/// convert an Unicode buffer into a WinAnsi (code page 1252) string
procedure UnicodeBufferToWinAnsi(source: PWideChar; out Dest: WinAnsiString);

{$ifdef UNICODE}
/// convert a Delphi 2009/2010 Unicode string into our UTF-8 string
function UnicodeStringToUtf8(const S: string): RawUTF8; inline;

/// convert a Delphi 2009/2010 Unicode string into a WinAnsi (code page 1252) string
function UnicodeStringToWinAnsi(const S: string): WinAnsiString; inline;

/// convert our UTF-8 encoded string into a Delphi 2009/2010 Unicode string
// - this function is the same as direct assignment, since RawUTF8=AnsiString(CP_UTF8),
// but is faster, since it uses no Win32 API call
function UTF8DecodeToUnicodeString(const S: RawUTF8): UnicodeString; overload; inline;

/// convert our UTF-8 encoded buffer into a Delphi 2009/2010 Unicode string
// - this function is the same as direct assignment, since RawUTF8=AnsiString(CP_UTF8),
// but is faster, since use no Win32 API call
function UTF8DecodeToUnicodeString(P: PUTF8Char; L: integer): UnicodeString; overload;
{$endif}

/// convert any generic VCL Text into an UTF-8 encoded String
// - it's prefered to use TLanguageFile.StringToUTF8() method in SQLite3i18n,
// which will handle full i18n of your application
// - it will work as is with Delphi 2009/2010 (direct unicode conversion)
// - under older version of Delphi (no unicode), it will use the
// current RTL codepage, as with WideString conversion (but without slow
// WideString usage)
function StringToUTF8(const Text: string): RawUTF8;

/// convert any UTF-8 encoded String into a generic VCL Text
// - it's prefered to use TLanguageFile.UTF8ToString() in SQLite3i18n,
// which will handle full i18n of your application
// - it will work as is with Delphi 2009/2010 (direct unicode conversion)
// - under older version of Delphi (no unicode), it will use the
// current RTL codepage, as with WideString conversion (but without slow
// WideString usage)
function UTF8ToString(const Text: RawUTF8): string;

/// fast Format() function replacement, optimized for RawUTF8
// - only supported token is %, which works only for integer and string type of
// const Args
// - resulting string has no length limit and uses fast concatenation
// - maximum count of supplied argument in Args is 10
function FormatUTF8(Format: PUTF8Char; const Args: array of const): RawUTF8;

/// convert an open array (const Args: array of const) argument to an UTF-8
// encoded text
procedure VarRecToUTF8(const V: TVarRec; var result: RawUTF8);


{$ifndef ENHANCEDRTL} { is our Enhanced Runtime (or LVCL) library not installed? }

{$define OWNNORMTOUPPER} { NormToUpper[] exists only in our enhanced RTL }

{$ifndef PUREPASCAL} { these functions are implemented in asm }
{$ifndef LVCL} { don't define these functions twice }
{$ifndef FPC}  { these asm function use some low-level system.pas calls }

/// use our fast asm RawUTF8 version of Trim()
function Trim(const S: RawUTF8): RawUTF8;

{$endif FPC}

{$ifdef UNICODE}
/// our fast RawUTF8 version of Pos(), for Unicode only compiler
// - this Pos() is seldom used, but this RawUTF8 specific version is needed
// by Delphi 2009/2010, to avoid two unnecessary conversions into UnicodeString
function Pos(const substr, str: RawUTF8): Integer; overload; inline;
{$endif}

{$endif LVCL}
{$endif PUREPASCAL}

/// use our fast RawUTF8 version of IntToStr()
// - without any slow UnicodeString=String->AnsiString conversion for Delphi 2009
// - only usefull if our Enhanced Runtime (or LVCL) library is not installed
function Int64ToUtf8(Value: Int64): RawUTF8;

/// use our fast RawUTF8 version of IntToStr()
// - without any slow UnicodeString=String->AnsiString conversion for Delphi 2009
// - only usefull if our Enhanced Runtime (or LVCL) library is not installed
function Int32ToUtf8(Value: integer): RawUTF8;

{$endif ENHANCEDRTL}

/// convert a floating-point value to its numerical text equivalency
// - returns the count of chars stored into S (S[0] is not set)
function ExtendedToString(var S: ShortString; Value: Extended; decimals: integer): integer;

/// convert a floating-point value to its numerical text equivalency
function ExtendedToStr(Value: Extended; decimals: integer): RawUTF8;

/// fast retrieve the position of a given character
function PosChar(Str: PUTF8Char; Chr: AnsiChar): PUTF8Char;

/// a non case-sensitive RawUTF8 version of Pos()
// - substr is expected to be already in upper case
function PosI(substr: PUTF8Char; const str: RawUTF8): Integer;

{{ internal fast integer val to text conversion
 - expect the last available temporary char position in P
 - return the last written char position (write in reverse order in P^)
 - typical use:
  !function Int32ToUTF8(Value : integer): RawUTF8;
  !var tmp: array[0..15] of AnsiChar;
  !    P: PAnsiChar;
  !begin
  !  P := StrInt32(@tmp[15],Value);
  !  SetString(result,P,@tmp[15]-P);
  !end;
 - not to be called directly: use IntToStr() instead }
function StrInt32(P: PAnsiChar; val: PtrInt): PAnsiChar;

{$ifndef CPU64} { StrInt32 aldready implemented PtrInt=Int64 }
{{ internal fast Int64 val to text conversion
   - same calling convention as with StrInt32() above }
function StrInt64(P: PAnsiChar; val: Int64): PAnsiChar;
{$endif}

/// fast add some characters to a RawUTF8 string
// - faster than SetString(tmp,Buffer,BufferLen); Text := Text+tmp;
procedure AppendBufferToRawUTF8(var Text: RawUTF8; Buffer: pointer; BufferLen: PtrInt);

/// use our fast version of StrComp(), to be used with PUTF8Char
function StrComp(Str1, Str2: PUTF8Char): integer;

/// use our fast version of StrIComp()
function StrIComp(Str1, Str2: PUTF8Char): integer;

/// our fast version of StrLen(), to be used with PUTF8Char
function StrLen(S: PUTF8Char): PtrInt;

/// our fast version of StrLen(), to be used with PWideChar
function StrLenW(S: PWideChar): PtrInt;

{$ifdef USENORMTOUPPER}
{$ifdef OWNNORMTOUPPER}
type
  TNormTable = packed array[AnsiChar] of AnsiChar;
  TNormTableByte = packed array[byte] of byte;

var
  /// the NormToUpper[] array is defined in our Enhanced RTL: define it now
  //  if it was not installed
  NormToUpper: TNormTable;
  NormToUpperByte: TNormTableByte absolute NormToUpper;

  /// the NormToLower[] array is defined in our Enhanced RTL: define it now
  //  if it was not installed
  NormToLower: TNormTable;
  NormToLowerByte: TNormTableByte absolute NormToLower;
{$endif}
{$else}
{$undef OWNNORMTOUPPER}
{$endif}

/// get the signed 32 bits integer value stored in P^
// - we use the PtrInt result type, even if expected to be 32 bits, to use
// native CPU register size (don't want any 32 bits overflow here)
function GetInteger(P: PUTF8Char): PtrInt; overload;

/// get the signed 32 bits integer value stored in P^
// - this version return 0 in err if no error occured, and 1 if an invalid
// character was found, not its exact index as for the val() function
// - we use the PtrInt result type, even if expected to be 32 bits, to use
// native CPU register size (don't want any 32 bits overflow here)
function GetInteger(P: PUTF8Char; var err: integer): PtrInt; overload;

/// get the unsigned 32 bits integer value stored in P^
// - we use the PtrInt result type, even if expected to be 32 bits, to use
// native CPU register size (don't want any 32 bits overflow here)
function GetCardinal(P: PUTF8Char): PtrUInt;

/// get the unsigned 32 bits integer value stored as Unicode string in P^
function GetCardinalW(P: PWideChar): PtrUInt;

/// get the 64 bits integer value stored in P^
function GetInt64(P: PUTF8Char): Int64; overload;
  {$ifdef CPU64}inline;{$endif}

/// get the 64 bits integer value stored in P^
// - set the err content to the index of any faulty character, 0 if conversion
// was successful (same as the standard val function)
function GetInt64(P: PUTF8Char; var err: integer): Int64; overload;

/// get the extended floating point value stored in P^
// - set the err content to the index of any faulty character, 0 if conversion
// was successful (same as the standard val function)
function GetExtended(P: PUTF8Char; var err: integer): extended;

/// get the WideChar stored in P^ (decode UTF-8 if necessary)
function GetUTF8Char(P: PUTF8Char): PtrUInt;

/// get the WideChar stored in P^ (decode UTF-8 if necessary) and set new pos to Next
function NextUTF8Char(P: PUTF8Char; out Next: PUTF8Char): PtrUInt;

/// encode a string to be compatible with URI encoding
function UrlEncode(const svar: RawUTF8): RawUTF8; overload;

/// encode supplied parameters to be compatible with URI encoding
// - parameters must be supplied two by two, as Name,Value pairs, e.g.
// ! url := UrlEncode(['select','*','where','ID=12','offset',23]);
// - parameters can be either textual, integer or extended
function UrlEncode(const NameValuePairs: array of const): RawUTF8; overload;

/// decode a string compatible with URI encoding into its original value
// - you can specify the decoding range (as in copy(s,i,len) function)
function UrlDecode(const s: RawUTF8; i: PtrInt = 1; len: PtrInt = -1): RawUTF8;

/// decode a specified parameter compatible with URI encoding into its original
// textual value
// - UrlDecodeValue('select=%2A&where=LastName%3D%27M%C3%B4net%27','SELECT=',V,@Next)
// will return Next^='where=...' and V='*'
// - if Upper is not found, Value is not modified, and result is FALSE
// - if Upper is found, Value is modified with the supplied content, and result is TRUE
function UrlDecodeValue(U, Upper: PUTF8Char; var Value: RawUTF8; Next: PPUTF8Char=nil): boolean;

/// decode a specified parameter compatible with URI encoding into its original
// numerical value
// - UrlDecodeInteger('offset=20&where=LastName%3D%27M%C3%B4net%27','OFFSET=',O,@Next)
// will return Next^='where=...' and O=20
// - if Upper is not found, Value is not modified, and result is FALSE
// - if Upper is found, Value is modified with the supplied content, and result is TRUE
function UrlDecodeInteger(U, Upper: PUTF8Char; var Value: integer; Next: PPUTF8Char=nil): boolean;

/// decode a specified parameter compatible with URI encoding into its original
// floating-point value
// - UrlDecodeExtended('price=20.45&where=LastName%3D%27M%C3%B4net%27','PRICE=',P,@Next)
// will return Next^='where=...' and P=20.45
// - if Upper is not found, Value is not modified, and result is FALSE
// - if Upper is found, Value is modified with the supplied content, and result is TRUE
function UrlDecodeExtended(U, Upper: PUTF8Char; var Value: Extended; Next: PPUTF8Char=nil): boolean;

/// returns TRUE if all supplied parameters does exist in the URI encoded text
// - UrlDecodeNeedParameters('price=20.45&where=LastName%3D','PRICE,WHERE') will
// return TRUE
function UrlDecodeNeedParameters(U, CSVUpper: PUTF8Char): boolean;

/// case unsensitive test of P1 and P2 content
// - use it with properties only (A..Z,0..9 chars)
function IdemPropName(const P1,P2: shortstring): boolean; overload;

/// case unsensitive test of P1 and P2 content
// - use it with properties only (A..Z,0..9 chars)
// - this version expect P2 to be a PAnsiChar with a specified length
function IdemPropName(const P1: shortstring; P2: PUTF8Char; P2Len: integer): boolean; overload;

/// returns true if the beginning of p^ is the same as up^
// - ignore case - up^ must be already Upper
// - chars are compared as WinAnsi (codepage 1252), not as UTF-8
function IdemPChar(p, up: PUTF8Char): boolean;

/// returns true if the file name extension contained in p^ is the same same as extup^
// - ignore case - extup^ must be already Upper
// - chars are compared as WinAnsi (codepage 1252), not as UTF-8
function IdemFileExt(p, extup: PUTF8Char): Boolean;

/// return true if up^ is contained inside the UTF-8 buffer p^
// - search up^ at the beginning of every UTF-8 word (aka in Soundex)
// - up^ must be already Upper
function ContainsUTF8(p, up: PUTF8Char): boolean;

/// returns true if the beginning of p^ is same as up^
// - ignore case - up^ must be already Upper
// - this version expect p^ to point to an Unicode char array
// - chars are compared as WinAnsi (codepage 1252), not as UTF-8
function IdemPCharW(p: pWideChar; up: PUTF8Char): boolean;

/// copy source into dest^ with upper case conversion
// - returns final dest pointer
// - will copy up to 255 AnsiChar (expect the dest buffer to be array[byte] of
// AnsiChar)
function UpperCopy255(dest: PAnsiChar; const source: RawUTF8): PAnsiChar;

/// copy source into dest^ with upper case conversion
// - returns final dest pointer
// - will copy up to 255 AnsiChar (expect the dest buffer to be array[byte] of
// AnsiChar)
function UpperCopy(dest: PAnsiChar; const source: RawUTF8): PAnsiChar;

/// copy source into dest^ with upper case conversion
// - returns final dest pointer
// - this special version expect source to be a shortstring
function UpperCopyShort(dest: PAnsiChar; const source: shortstring): PAnsiChar;

{$ifdef USENORMTOUPPER}
/// fast UTF-8 comparaison using the NormToUpper[] array for all 8 bits values
// - this version expects u1 and u2 to be zero-terminated
function UTF8IComp(u1, u2: PUTF8Char): PtrInt;

/// fast UTF-8 comparaison using the NormToUpper[] array for all 8 bits values
// - this version expects u1 and u2 not to be necessary zero-terminated, but
// uses L1 and L2 as length for u1 and u2 respectively
// - use this function for SQLite3 collation (TSQLCollateFunc)
function UTF8ILComp(u1, u2: PUTF8Char; L1,L2: cardinal): PtrInt;

/// fast Unicode comparaison using the NormToUpper[] array for all 8 bits values
// - this version expects u1 and u2 to be zero-terminated
function AnsiICompW(u1, u2: PWideChar): PtrInt;

/// SameText() overloaded function with proper UTF-8 decoding
// - fast version using NormtoUpper[] array for all Win-Ansi characters
function SameTextU(const S1, S2: RawUTF8): Boolean;
  {$ifdef PUREPASCAL} {$ifdef HASINLINE}inline;{$endif} {$endif}

{$endif}
/// fast WinAnsi comparaison using the NormToUpper[] array for all 8 bits values
function AnsiIComp(Str1, Str2: PUTF8Char): PtrInt;
  {$ifndef USENORMTOUPPER} {$ifdef PUREPASCAL}
  {$ifdef HASINLINE}inline;{$endif} {$endif} {$endif}

/// extract a line from source array of chars
// - next will contain the beginning of next line, or nil if source if ended
function GetNextLine(source: PUTF8Char; out next: PUTF8Char): RawUTF8;

{$ifdef UNICODE}
/// extract a line from source array of chars
// - next will contain the beginning of next line, or nil if source if ended
// - this special version expect UnicodeString pointers, and return an UnicodeString
function GetNextLineW(source: PWideChar; out next: PWideChar): string;

/// find the Value of UpperName in P, till end of current section
// - expect UpperName as 'NAME='
// - this special version expect UnicodeString pointer, and return an UnicodeString
function FindIniNameValueW(P: PWideChar; UpperName: PUTF8Char): string;

/// find a Name= Value in a [Section] of a INI Unicode Content
// - this function scans the Content memory buffer, and is
// therefore very fast (no temporary TMemIniFile is created)
// - if Section equals '', find the Name= value before any [Section]
function FindIniEntryW(const Content: string; const Section, Name: RawUTF8): string;

{$ifdef PUREPASCAL}

/// our fast RawUTF8 version of Trim(), for Unicode only compiler
// - this Trim() is seldom used, but this RawUTF8 specific version is needed
// by Delphi 2009/2010, to avoid two unnecessary conversions into UnicodeString
function Trim(const S: RawUTF8): RawUTF8;

/// our fast RawUTF8 version of Pos(), for Unicode only compiler
// - this Pos() is seldom used, but this RawUTF8 specific version is needed
// by Delphi 2009/2010, to avoid two unnecessary conversions into UnicodeString
function Pos(const substr, str: RawUTF8): Integer; overload; inline;

{$endif}

{$endif}

/// return true if IdemPChar(source,search), and go to the next line of source
function IdemPCharAndGetNextLine(var source: PUTF8Char; search: PUTF8Char): boolean;

/// return line begin from source array of chars, and go to next line
// - next will contain the beginning of next line, or nil if source if ended
function GetNextLineBegin(source: PUTF8Char; out next: PUTF8Char): PUTF8Char;

/// return next CSV string from P, nil if no more
function GetNextItem(var P: PUTF8Char; Sep: AnsiChar= ','): RawUTF8;

/// return next CSV string from P, nil if no more
// - this function return the generic string type of the compiler, and
// therefore can be used with ready to be displayed text (e.g. for the VCL)
function GetNextItemString(var P: PChar; Sep: Char= ','): string;

/// append some text lines with the supplied Values[]
// - if any Values[] item is '', no line is added
// - otherwize, appends 'Caption: Value', with Caption taken from CSV
procedure AppendCSVValues(const CSV: string; const Values: array of string;
  var Result: string; const AppendBefore: string=#13#10);

/// return next CSV string from P, nil if no more
procedure GetNextItemShortString(var P: PUTF8Char; out Dest: ShortString; Sep: AnsiChar= ',');

/// return next CSV string as unsigned integer from P, 0 if no more
function GetNextItemCardinal(var P: PUTF8Char; Sep: AnsiChar= ','): cardinal;

/// return next CSV string as unsigned integer from P, 0 if no more
// - this version expect P^ to point to an Unicode char array
function GetNextItemCardinalW(var P: PWideChar; Sep: WideChar= ','): Cardinal;

/// return next CSV string as double from P, 0.0 if no more
function GetNextItemDouble(var P: PUTF8Char; Sep: AnsiChar= ','): double;

/// return n-th indexed CSV string in P, starting at Index=0 for first one
function GetCSVItem(P: PUTF8Char; Index: PtrUInt; Sep: AnsiChar = ','): RawUTF8;

/// return n-th indexed CSV string in P, starting at Index=0 for first one
// - this function return the generic string type of the compiler, and
// therefore can be used with ready to be displayed text (i.e. the VCL)
function GetCSVItemString(P: PChar; Index: PtrUInt; Sep: Char = ','): string;

/// return the index of a Value in a CSV string
// - start at Index=0 for first one
// - return -1 if specified Value was not found in CSV items
function FindCSVIndex(CSV: PUTF8Char; const Value: RawUTF8; Sep: AnsiChar = ',';
  CaseSensitive: boolean=true): integer;

/// add the strings in the specified CSV text into a dynamic array of strings
procedure CSVToRawUTF8DynArray(CSV: PUTF8Char; var Result: TRawUTF8DynArray;
  Sep: AnsiChar = ',');

/// return the index of Value in Values[], -1 if not found
function FindRawUTF8(const Values: TRawUTF8DynArray; const Value: RawUTF8;
  CaseSensitive: boolean=true): integer;

/// true if Value was added successfully in Values[]
function AddRawUTF8(var Values: TRawUTF8DynArray; const Value: RawUTF8;
  NoDupplicates: boolean=false; CaseSensitive: boolean=true): boolean;

/// find a Name= Value in a [Section] of a INI RawUTF8 Content
// - this function scans the Content memory buffer, and is
// therefore very fast (no temporary TMemIniFile is created)
// - if Section equals '', find the Name= value before any [Section]
function FindIniEntry(const Content, Section,Name: RawUTF8): RawUTF8;

/// find a Name= Value in a [Section] of a INI WinAnsi Content
// - same as FindIniEntry(), but the value is converted from WinAnsi into UTF-8
function FindWinAnsiIniEntry(const Content, Section,Name: RawUTF8): RawUTF8;

/// find a Name= numeric Value in a [Section] of a INI RawUTF8 Content and
// return it as an integer, or 0 if not found
// - this function scans the Content memory buffer, and is
// therefore very fast (no temporary TMemIniFile is created)
// - if Section equals '', find the Name= value before any [Section]
function FindIniEntryInteger(const Content, Section,Name: RawUTF8): integer;
  {$ifdef HASINLINE}inline;{$endif}

/// find a Name= Value in a [Section] of a .INI file
// - if Section equals '', find the Name= value before any [Section]
// - use internaly fast FindIniEntry() function above
function FindIniEntryFile(const FileName: TFileName; const Section,Name: RawUTF8): RawUTF8;

/// update a Name= Value in a [Section] of a INI RawUTF8 Content
// - this function scans and update the Content memory buffer, and is
// therefore very fast (no temporary TMemIniFile is created)
// - if Section equals '', update the Name= value before any [Section]
procedure UpdateIniEntry(var Content: RawUTF8; const Section,Name,Value: RawUTF8);

/// update a Name= Value in a [Section] of a .INI file
// - if Section equals '', update the Name= value before any [Section]
// - use internaly fast UpdateIniEntry() function above
procedure UpdateIniEntryFile(const FileName: TFileName; const Section,Name,Value: RawUTF8);

/// find the position of the [SEARCH] section in source
// - return true if [SEARCH] was found, and store pointer to the line after it in source
function FindSectionFirstLine(var source: PUTF8Char; search: PUTF8Char): boolean;

/// find the position of the [SEARCH] section in source
// - return true if [SEARCH] was found, and store pointer to the line after it in source
// - this version expect source^ to point to an Unicode char array
function FindSectionFirstLineW(var source: PWideChar; search: PUTF8Char): boolean;

/// retrieve the whole content of a section as a string
// - SectionFirstLine may have been obtained by FindSectionFirstLine() function above
function GetSectionContent(SectionFirstLine: PUTF8Char): RawUTF8; overload;

/// retrieve the whole content of a section as a string
// - use SectionFirstLine() then previous GetSectionContent()
function GetSectionContent(const Content, SectionName: RawUTF8): RawUTF8; overload;

/// delete a whole [Section]
// - if EraseSectionHeader is TRUE (default), then the [Section] line is also
// deleted together with its content lines
// - return TRUE if something was changed in Content
// - return FALSE if [Section] doesn't exist or is already void
function DeleteSection(var Content: RawUTF8; const SectionName: RawUTF8;
  EraseSectionHeader: boolean=true): boolean; overload;

/// delete a whole [Section]
// - if EraseSectionHeader is TRUE (default), then the [Section] line is also
// deleted together with its content lines
// - return TRUE if something was changed in Content
// - return FALSE if [Section] doesn't exist or is already void
// - SectionFirstLine may have been obtained by FindSectionFirstLine() function above
function DeleteSection(SectionFirstLine: PUTF8Char; var Content: RawUTF8;
  EraseSectionHeader: boolean=true): boolean; overload;

/// replace a whole [Section] content by a new content
// - create a new [Section] if none was existing
procedure ReplaceSection(var Content: RawUTF8; const SectionName,
  NewSectionContent: RawUTF8); overload;

/// replace a whole [Section] content by a new content
// - create a new [Section] if none was existing
// - SectionFirstLine may have been obtained by FindSectionFirstLine() function above
procedure ReplaceSection(SectionFirstLine: PUTF8Char;
  var Content: RawUTF8; const NewSectionContent: RawUTF8); overload;

/// return TRUE if Value of UpperName does exist in P, till end of current section
// - expect UpperName as 'NAME='
function ExistsIniName(P: PUTF8Char; UpperName: PUTF8Char): boolean;

/// find the Value of UpperName in P, till end of current section
// - expect UpperName as 'NAME='
function FindIniNameValue(P: PUTF8Char; UpperName: PUTF8Char): RawUTF8;

/// return TRUE if the Value of UpperName exists in P, till end of current section
// - expect UpperName as 'NAME='
function ExistsIniNameValue(P: PUTF8Char; UpperName: PUTF8Char): boolean;

/// find the Value of UpperName in P, till end of current section
// - expect UpperName as 'NAME='
// - same as FindIniNameValue(), but the value is converted from WinAnsi into UTF-8
function FindWinAnsiIniNameValue(P: PUTF8Char; UpperName: PUTF8Char): RawUTF8;

/// find the integer Value of UpperName in P, till end of current section
// - expect UpperName as 'NAME='
// - return 0 if no NAME= entry was found
function FindIniNameValueInteger(P: PUTF8Char; UpperName: PUTF8Char): integer;
  {$ifdef HASINLINE}inline;{$endif}

/// read a File content into a String
// - content can be binary or text
// - returns '' if file was not found or any read error occured
// - uses RawByteString for byte storage, thatever the codepage is
function StringFromFile(const FileName: TFileName): RawByteString;

/// create a File from a string content
// - uses RawByteString for byte storage, thatever the codepage is
function FileFromString(const Content: RawByteString; const FileName: TFileName;
  FlushOnDisk: boolean=false): boolean;

/// get the file date and time
// - return 0 if file doesn't exist
function FileAgeToDateTime(const FileName: TFileName): TDateTime;

/// extract file name, without its extension
function GetFileNameWithoutExt(const FileName: TFileName): TFileName;

/// retrieve a property value in a text-encoded class
// - follows the Delphi serialized text object format, not standard .ini
// - if the property is a string, the simple quotes ' are trimed
function FindObjectEntry(const Content, Name: RawUTF8): RawUTF8;

/// retrieve a filename property value in a text-encoded class
// - follows the Delphi serialized text object format, not standard .ini
// - if the property is a string, the simple quotes ' are trimed
// - any file path and any extension are trimmed
function FindObjectEntryWithoutExt(const Content, Name: RawUTF8): RawUTF8;


type
  {{ available pronunciations for our fast Soundex implementation }
  TSynSoundExPronunciation =
    (sndxEnglish, sndxFrench, sndxSpanish, sndxNone);

  TSoundExValues = array[0..ord('Z')-ord('B')] of byte;
  PSoundExValues = ^TSoundExValues;

  PSynSoundEx = ^TSynSoundEx;
  {{ fast search of a text value, using the Soundex searching mechanism
    - Soundex is a phonetic algorithm for indexing names by sound,
      as pronounced in a given language. The goal is for homophones to be
      encoded to the same representation so that they can be matched despite
      minor differences in spelling
    - this implementation is very fast and can be used e.g. to parse and search
      in a huge text buffer
    - This version also handles french and spanish pronunciations on request,
      which differs from default Soundex, i.e. English }
  TSynSoundEx =  object
  private
    Search, FirstChar: cardinal;
    fValues: PSoundExValues;
  public
    /// prepare for a Soundex search
    // - you can specify another language pronunciation than default english
    function Prepare(UpperValue: PAnsiChar;
      Lang: TSynSoundExPronunciation=sndxEnglish): boolean;
    /// return true if prepared value is contained in a text buffer
    // (UTF-8 encoded), by using the SoundEx comparison algorithm
    // - search prepared value at every word beginning in U^
    function UTF8(U: PUTF8Char): boolean;
    /// return true if prepared value is contained in a ANSI text buffer
    // by using the SoundEx comparison algorithm
    // - search prepared value at every word beginning in A^
    function Ansi(A: PAnsiChar): boolean;
  end;

{{ Retrieve the Soundex value of a text word, from Ansi buffer
  - Return the soundex value as an easy to use cardinal value, 0 if the
    incoming string contains no valid word
  - if next is defined, its value is set to the end of the encoded word
    (so that you can call again this function to encode a full sentence) }
function SoundExAnsi(A: PAnsiChar; next: PPAnsiChar=nil;
  Lang: TSynSoundExPronunciation=sndxEnglish): cardinal;

{{ Retrieve the Soundex value of a text word, from UTF-8 buffer
  - Return the soundex value as an easy to use cardinal value, 0 if the
    incoming string contains no valid word
  - if next is defined, its value is set to the end of the encoded word
    (so that you can call again this function to encode a full sentence)
  - very fast: all UTF-8 decoding is handled on the fly }
function SoundExUTF8(U: PUTF8Char; next: PPUTF8Char=nil;
  Lang: TSynSoundExPronunciation=sndxEnglish): cardinal;

const
  /// number of bits to use for each interresting soundex char
  // - default is to use 8 bits, i.e. 4 soundex chars, which is the
  // standard approach
  // - for a more detailled soundex, use 4 bits resolution, which will
  // compute up to 7 soundex chars in a cardinal (that's our choice)
  SOUNDEX_BITS = 4;

/// return true if UpperValue (Ansi) is contained in A^ (Ansi)
// - find UpperValue starting at word beginning, not inside words
function FindAnsi(A, UpperValue: PAnsiChar): boolean;

/// return true if UpperValue (Ansi) is contained in U^ (UTF-8 encoded)
// - find UpperValue starting at word beginning, not inside words
// - UTF-8 decoding is done on the fly (no temporary decoding buffer is used)
function FindUTF8(U: PUTF8Char; UpperValue: PAnsiChar): boolean;

/// trim first lowercase chars ('otDone' will return 'Done' e.g.)
// - return an RawUTF8 string: enumeration names are pure 7bit ANSI with Delphi 7
// to 2007, and UTF-8 encoded with Delphi 2009/2010
function TrimLeftLowerCase(V: PShortString): RawUTF8;

/// convert a CamelCase string into a space separated one
// - 'OnLine' will return 'On line' e.g., and 'OnMyLINE' will return 'On my LINE'
// - '_' char is transformed into ' - '
// - '__' chars are transformed into ': '
// - return an RawUTF8 string: enumeration names are pure 7bit ANSI with Delphi 7
// to 2007, and UTF-8 encoded with Delphi 2009/2010
function UnCamelCase(const S: RawUTF8): RawUTF8; overload;

/// convert a CamelCase string into a space separated one
// - 'OnLine' will return 'On line' e.g., and 'OnMyLINE' will return 'On my LINE'
// - return the char count written into D^
// - D^ and P^ are expected to be UTF-8 encoded: enumeration and property names
// are pure 7bit ANSI with Delphi 7 to 2007, and UTF-8 encoded with Delphi 2009/2010
// - '_' char is transformed into ' - '
// - '__' chars are transformed into ': '
function UnCamelCase(D, P: PUTF8Char): integer; overload;

const
  /// US English Windows Code Page, i.e. WinAnsi standard character encoding
  CODEPAGE_US = 1252;

/// convert a char set to a code page
function CharSetToCodePage(CharSet: integer): cardinal;

/// convert a code page to a char set
function CodePageToCharSet(CodePage: Cardinal): Integer;

/// retrieve the MIME content type from a supplied binary buffer
// - return the MIME type, ready to be appended to a 'Content-Type: ' HTTP header
// - default is 'application/octet-stream'
// - see @http://en.wikipedia.org/wiki/Internet_media_type for most common values
function GetMimeContentType(Content: Pointer; Len: integer): RawUTF8;


type
  PIntegerDynArray = ^TIntegerDynArray;
  TIntegerDynArray = array of integer;
  PInt64DynArray = ^TInt64DynArray;
  TInt64DynArray = array of Int64;

/// fast search of an unsigned integer position in an integer array
// - Count is the number of cardinal entries in P^
// - returns P where P^=Value
// - returns nil if Value was not found
function IntegerScan(P: PCardinalArray; Count: PtrInt; Value: cardinal): PCardinal;

/// fast search of an unsigned integer position in an integer array
// - returns true if P^=Value within Count entries
// - returns false if Value was not found
function IntegerScanExists(P: PCardinalArray; Count: PtrInt; Value: cardinal): boolean;

/// fast search of an unsigned integer position in an integer array
// - Count is the number of integer entries in P^
// - return index of P^[index]=Value
// - return -1 if Value was not found
function IntegerScanIndex(P: PCardinalArray; Count: PtrInt; Value: cardinal): PtrInt;

/// sort an Integer array, low values first
procedure QuickSortInteger(ID: PIntegerArray; L, R: PtrInt);

/// fast binary search of an integer value in a sorted integer array
// - R is the last index of available integer entries in P^ (i.e. Count-1)
// - return index of P^[index]=Value
// - return -1 if Value was not found
function FastFindIntegerSorted(P: PIntegerArray; R: PtrInt; Value: integer): PtrInt;

/// true if Value was added successfully in Values[]
function AddInteger(var Values: TIntegerDynArray; Value: integer;
  NoDupplicates: boolean=false): boolean;

/// delete any integer in Values[]
procedure DeleteInteger(var Values: TIntegerDynArray; Index: PtrInt);

/// add the strings in the specified CSV text into a dynamic array of integer
procedure CSVToIntegerDynArray(CSV: PUTF8Char; var Result: TIntegerDynArray);

/// return the corresponding CSV text from a dynamic array of integer
function IntegerDynArrayToCSV(var Values: TIntegerDynArray): RawUTF8;

type
  TWordDynArray = array of word;
  PWordDynArray = ^TWordDynArray;

  /// used to store and retrieve Words in a sorted array
  TSortedWordArray = object
    Values: TWordDynArray;
    Count: integer;
    /// add a value into the sorted array
    // - return the index of the new inserted value into the Values[] array
    // - return -(foundindex+1) if this value is already in the Values[] array
    function Add(aValue: Word): PtrInt;
    /// return the index if the supplied value in the Values[] array
    // - return -1 if not found
    function IndexOf(aValue: Word): PtrInt; {$ifdef HASINLINE}inline;{$endif}
  end;


{ ****************** text buffer and JSON functions and classes ********* }

type
  /// simple writer to a Stream, specialized for the TEXT format
  // - use an internal buffer, faster than string+string
  // - some dedicated methods is able to encode any data with JSON escape
  TTextWriter = class
  protected
    B, BEnd: PUTF8Char;
    fStream: TStream;
    fInitialStreamPosition: integer;
    fStreamIsOwned: boolean;
    /// flush and go to next char
    procedure FlushInc;
    function GetLength: integer;
  public
    /// the data will be written to the specified Stream
    constructor Create(aStream: TStream);
    /// the data will be written to an internal TMemoryStream
    constructor CreateOwnedStream;
    /// release fStream is is owned
    destructor Destroy; override;
    /// retrieve the data as a string
    // - only works if the associated Stream Inherits from TMemoryStream: return
    // '' if it is not the case
    function Text: RawUTF8;
    /// write pending data to the Stream
    procedure Flush;
    /// append one char to the buffer
    procedure Add(c: AnsiChar); overload; {$ifdef HASINLINE}inline;{$endif}
    /// append an Integer Value as a String
    procedure Add(Value: Int64); overload;
    /// append an Integer Value as a String
    procedure Add(Value: integer); overload;
    /// append a floating-point Value as a String
    // - double precision with max 3 decimals is default here, to avoid rounding
    // problems
    procedure Add(Value: double; decimals: integer=3); overload;
    /// append strings or integers with a specified format
    // - % = #37 indicates a string or integer parameter
    // - $ = #36 indicates an integer to be written with 2 digits and a comma
    // -  = #163 indicates an integer to be written with 4 digits and a comma
    // -  = #181 indicates an integer to be written with 3 digits without any comma
    // -  = #164 indicates CR+LF chars
    // - CR = #13 indicates CR+LF chars
    // -  = #167 indicates to trim last comma
    // - since some of this characters above are > #127, they are not UTF-8
    // ready, so we expect the input format to be WinAnsi, i.e. mostly English
    // text (with chars < #128) with some values to be inserted inside
    procedure Add(Format: PWinAnsiChar; const Values: array of const); overload;
    /// append CR+LF chars
    procedure AddCR; {$ifdef HASINLINE}inline;{$endif}
    /// append an Integer Value as a 2 digits String with comma
    procedure Add2(Value: integer);
    /// append an Integer Value as a 4 digits String with comma
    procedure Add4(Value: integer);
    /// append an Integer Value as a 3 digits String without any added comma
    procedure Add3(Value: integer);
    /// append a line of text with CR+LF at the end
    procedure AddLine(const Text: shortstring);
    /// append a String
    procedure AddString(const Text: RawUTF8); {$ifdef HASINLINE}inline;{$endif}
    /// append an array of integers as CSV
    procedure AddCSVIntegers(const Integers: array of Integer);
    /// write some data as hexa chars
    procedure WrHex(P: PAnsiChar; Len: integer);
    /// append some chars to the buffer
    // - if Len is 0, Len is calculated from zero-ended char
    // - don't escapes chars according to the JSON RFC
    procedure AddNoJSONEscape(P: Pointer; Len: integer=0);
    /// append some unicode chars to the buffer
    // - WideCharCount is the unicode chars count, not the byte size
    // - don't escapes chars according to the JSON RFC
    // - will convert the Unicode chars into UTF-8
    procedure AddNoJSONEscapeW(P: Pointer; WideCharCount: integer=0);
    /// append some UTF-8 encoded chars to the buffer
    // - if Len is 0, Len is calculated from zero-ended char
    // - escapes chars according to the JSON RFC
    procedure AddJSONEscape(P: Pointer; Len: PtrInt=0); overload;
    /// append some Unicode encoded chars to the buffer
    // - if Len is 0, Len is calculated from zero-ended widechar
    // - escapes chars according to the JSON RFC
    procedure AddJSONEscapeUnicode(P: PWord; Len: PtrInt=0);
    /// append an open array constant value to the buffer
    // - "" will be added if necessary
    // - escapes chars according to the JSON RFC
    // - very fast (avoid most temporary storage)
    procedure AddJSONEscape(const V: TVarRec); overload;
    /// the last char appended is canceled
    procedure CancelLastChar; {$ifdef HASINLINE}inline;{$endif}
    /// the last char appended is canceled if it was a ','
    procedure CancelLastComma; {$ifdef HASINLINE}inline;{$endif}
    /// rewind the Stream to the position when Create() was called
    procedure CancelAll;
    /// count of add byte to the stream
    property TextLength: integer read GetLength;
  protected // private arrays defined at the end of the class, for shorter code
    // [0..4] for 'u0001' four-hex-digits template, [5..7] for one UTF-8 char
    BufUnicode: array[0..7] of AnsiChar;
    // 512 bytes internal temporary buffer
    buf: array[0..511] of AnsiChar;
  end;


/// encode the supplied data as an UTF-8 valid JSON content
// - data must be supplied two by two, as Name,Value pairs, e.g.
// ! JSONEncode(['name','John','year',1972])
// - you can specify a memory stream (for example any TSQLRest.TempMemoryStream)
// to make it faster, since no memory allocation will be processed
function JSONEncode(const NameValuePairs: array of const;
  TempMemoryStream: TMemoryStream=nil): RawUTF8;

/// decode the supplied UTF-8 JSON content for the supplied names
// - data will be set in Values, according to the Names supplied e.g.
// ! JSONDecode(JSON,['NAME','YEAR'],Values) -> Values[0]^='John'; Values[1]^='1972';
// - Names must be UPPERCASE
// - if any supplied name wasn't found its corresponding Values[] will be nil
// - this procedure will decode the JSON content in-memory, i.e. the PUtf8Char
// array is created inside JSON, which is therefore modified: make a private
// copy first if you want to reuse the JSON content
procedure JSONDecode(var JSON: RawUTF8;
  const UpperNames: array of PUTF8Char; var Values: TPUtf8CharDynArray); overload;

/// decode the supplied UTF-8 JSON content for the one supplied name
// - this procedure will decode the JSON content in-memory, so must be called
// only once with the same JSON data
function JSONDecode(var JSON: RawUTF8; const aUpperName: RawUTF8='RESULT'): RawUTF8; overload;

/// decode a JSON field in an UTF-8 encoded buffer (used in TSQLTableJSON.Create)
// - this function decodes in the P^ buffer memory itself (no memory allocation
// or copy), for faster process - so take care that it's an unique string
// - PDest points to the next field to be decoded, or nil on any unexpected end
// - null is decoded as nil
// - '"strings"' are decoded as 'strings'
// - strings are JSON unescaped (and \u0123 is converted to UTF-8 chars)
// - any integer value is left as its ascii representation
// - wasString is set to true if the JSON value was a "string"
// - works for both field names or values (e.g. '"FieldName":' or 'Value,')
// - EndOfObject (if not nil) is set to the JSON value char (',' ':' or '}' e.g.)
function GetJSONField(P: PUTF8Char; out PDest: PUTF8Char;
  wasString: PBoolean=nil; EndOfObject: PUTF8Char=nil): PUTF8Char;

/// true if P^ is a "string" value nor null/false/tru nor any numeric
// (floating or integer), according to the JSON encoding schema
function IsJSONString(P: PUTF8Char): boolean;


{ ************ some other common types and conversion routines }


/// convert a string into its INTEGER Curr64 (value*10000) representation
// - this type is compatible with Delphi currency memory map with PInt64(@Curr)^
// - fast conversion, using only integer operations
function StrToCurr64(P: PUTF8Char): Int64;

/// convert an INTEGER Curr64 (value*10000) into a string
// - this type is compatible with Delphi currency memory map with PInt64(@Curr)^
// - fast conversion, using only integer operations
// - decimals are joined by 2 (no decimal, 2 decimals, 4 decimals)
function Curr64ToStr(Value: Int64): RawUTF8;

/// convert an INTEGER Curr64 (value*10000) into a string
// - this type is compatible with Delphi currency memory map with PInt64(@Curr)^
// - fast conversion, using only integer operations
// - decimals are joined by 2 (no decimal, 2 decimals, 4 decimals)
// - return the number of chars written to Dest^
function Curr64ToPChar(Value: Int64; Dest: PUTF8Char): PtrInt;


var
  /// a conversion table from hexa chars into binary data
  // - use HexToBin() function 
  ConvertHexToBin: array of byte;

/// if any function use the ConvertHexToBin[] array, it must initialize it with
// this function if not yet available
procedure InitConvertHexToBin;

/// fast conversion from hexa chars into binary data
// - BinBytes contain the bytes count to be converted: Hex^ must contain
//  at least BinBytes*2 chars to be converted, and Bin^ enough space
// - if Bin=nil, no output data is written, but the Hex^ format is checked
// - return false if any invalid (non hexa) char is found in Hex^
// - using this function with Bin^ as an integer value will decode in big-endian
// order (most-signignifican byte first)
function HexToBin(Hex: PAnsiChar; Bin: PByte; BinBytes: Integer): boolean;

/// fast conversion from binary data into hexa chars
// - BinBytes contain the bytes count to be converted: Hex^ must contain
// enough space for at least BinBytes*2 chars
// - using this function with BinBytes^ as an integer value will encode it
// in big-endian order (most-signignifican byte first): don't use it for display
procedure BinToHex(Bin, Hex: PAnsiChar; BinBytes: integer); overload;

/// fast conversion from binary data into hexa chars
function BinToHex(const Bin: RawByteString): RawUTF8; overload;

/// add the 4 digits of integer Y to P^
procedure YearToPChar(Y: Word; P: PUTF8Char);
  {$ifdef PUREPASCAL} {$ifdef HASINLINE}inline;{$endif} {$endif}

/// compare to floating point values, with IEEE 754 double precision
// - use this function instead of raw = operator
// - the precision is calculated from the A and B value range
// - faster equivalent than SameValue() in Math unit
// - if you know the precision range of A and B, it's faster to check abs(A-B)<range
function SameValue(const A, B: Double; DoublePrec: double = 1E-10): Boolean;

// our custom hash function, specialized for Text comparaison
// - has less colision than Adler32 for short strings
// - is faster than CRC32 or Adler32, since use DQWord (128 bytes) aligned read
// - uses RawByteString for binary content hashing, thatever the codepage is
function Hash32(const Text: RawByteString): cardinal; overload;

// our custom hash function, specialized for Text comparaison
// - has less colision than Adler32 for short strings
// - is faster than CRC32 or Adler32, since use DQWord (128 bytes) aligned read
// - overloaded version for direct binary content hashing
function Hash32(Data: pointer; Len: integer): cardinal; overload;

/// retrieve a particular bit status from a bit array
function GetBit(const Bits; aIndex: PtrInt): boolean;
  {$ifdef PUREPASCAL} {$ifdef HASINLINE}inline;{$endif} {$endif}

/// set a particular bit into a bit array
procedure SetBit(var Bits; aIndex: PtrInt);
  {$ifdef PUREPASCAL} {$ifdef HASINLINE}inline;{$endif} {$endif}

/// unset/clear a particular bit into a bit array
procedure UnSetBit(var Bits; aIndex: PtrInt);
  {$ifdef PUREPASCAL} {$ifdef HASINLINE}inline;{$endif} {$endif}

/// compute the number of bits set in a bit array
// - Count is the bit count, not byte size
function GetBitsCount(const Bits; Count: PtrInt): integer;

/// retrieve a particular bit status from a Int64 bit array (max aIndex is 63)
function GetBit64(const Bits; aIndex: PtrInt): boolean;
  {$ifdef PUREPASCAL} {$ifdef HASINLINE}inline;{$endif} {$endif}

/// set a particular bit into a Int64 bit array (max aIndex is 63)
procedure SetBit64(var Bits: Int64; aIndex: PtrInt);
  {$ifdef PUREPASCAL} {$ifdef HASINLINE}inline;{$endif} {$endif}

/// unset/clear a particular bit into a Int64 bit array (max aIndex is 63)
procedure UnSetBit64(var Bits: Int64; aIndex: PtrInt);
  {$ifdef PUREPASCAL} {$ifdef HASINLINE}inline;{$endif} {$endif}

{$ifndef ENHANCEDRTL}
{$ifndef LVCL} { don't define these const twice }

const
  /// fast lookup table for converting any decimal number from
  // 0 to 99 into their ASCII equivalence
  // - our enhanced SysUtils.pas (normal and LVCL) contains the same array
  TwoDigitLookup: packed array[0..99] of array[1..2] of AnsiChar =
    ('00','01','02','03','04','05','06','07','08','09',
     '10','11','12','13','14','15','16','17','18','19',
     '20','21','22','23','24','25','26','27','28','29',
     '30','31','32','33','34','35','36','37','38','39',
     '40','41','42','43','44','45','46','47','48','49',
     '50','51','52','53','54','55','56','57','58','59',
     '60','61','62','63','64','65','66','67','68','69',
     '70','71','72','73','74','75','76','77','78','79',
     '80','81','82','83','84','85','86','87','88','89',
     '90','91','92','93','94','95','96','97','98','99');

{$endif}
{$endif}

var
  /// fast lookup table for converting any decimal number from
  // 0 to 99 into their ASCII equivalence
  TwoDigitLookupW: packed array[0..99] of word absolute TwoDigitLookup;

const
  /// used internaly for fast word recognition (32 bytes const)
  IsWord: set of byte =
    [ord('0')..ord('9'),ord('a')..ord('z'),ord('A')..ord('Z')];


{$ifdef MSWINDOWS}

type
  /// used to retrieve version information from any EXE
  TFileVersion = object
    Major: Integer;
    Minor: Integer;
    Release: Integer;
    Build: Integer;
    /// version info of the exe file as '3.1'
    // - return "string" type, i.e. UnicodeString for Delphi 2009/2010
    Main: string;
    /// version info of the exe file as '3.1.0.123'
    // - return "string" type, i.e. UnicodeString for Delphi 2009/2010
    Detailed: string;
    /// build date and time of this exe file
    BuildDateTime: TDateTime;
    /// build year of this exe file
    BuildYear: integer;
    /// retrieve the version as a 32 bits integer with Major.Minor.Release
    function Version32: integer;
    /// retrieve application version from exe file name
    // - DefaultVersion is used if no information Version was included into
    // the executable resources (on compilation time)
    // - to retrieve version information from current executable, just call e.g.
    // ApplicationVersion.RetrieveApplicationVersion(ParamStr(0),3)
    procedure RetrieveApplicationVersion(const FileName: TFileName; DefaultVersion: integer);
  end;

{$else}

/// compatibility function, to be implemented according to the running OS
// - expect more or less the same result as the homonymous Win32 API function
function GetTickCount: Cardinal;

{$endif MSWINDOWS}

type
  /// to be used instead of TMemoryStream, for speed
  // - allocates memory from Delphi heap (i.e. FastMM4) and not GlobalAlloc()
  // - uses bigger growing size of the capacity
{$ifdef LVCL} // LVCL already use Delphi heap instead of GlobalAlloc()
  THeapMemoryStream = TMemoryStream;
{$else}
  THeapMemoryStream = class(TMemoryStream)
  protected
    function Realloc(var NewCapacity: Longint): Pointer; override;
  end;
{$endif}


{ ************ Unit-Testing classes and functions }

type
{$ifdef MSWINDOWS}
  /// high resolution timer (for accurate speed statistics)
  TPrecisionTimer = object
  private
    iStart,iStop: Int64;
    /// contains the time elapsed in micro seconds between Start and Stop
    iTime: Int64;
  public
    /// start the high resolution timer
    procedure Start;
    /// stop the timer, returning the time elapsed, with appened time resolution (us,ms,s)
    function Stop: RawUTF8;
    /// return the time elapsed, with appened time resolution (us,ms,s)
    function Time: RawUTF8;
    /// compute the per second count
    function PerSec(Count: cardinal): cardinal;
    /// compute the time elapsed by count, with appened time resolution (us,ms,s)
    function ByCount(Count: cardinal): RawUTF8;
  end;
{$endif}

  /// the prototype of an individual test
  // - to be used with TSynTest descendants
  TSynTestEvent = procedure of object;

{$M+} { we need the RTTI for the published methods of this object class }
  /// a generic class for both tests suit and cases
  // - purpose of this ancestor is to have RTTI for its published methods,
  // and to handle a class text identifier, or uncamelcase its class name
  // if no identifier was defiend
  TSynTest = class
  protected
    fTests: array of record
      TestName: string;
      Method: TSynTestEvent;
    end;
    fIdent: string;
    fInternalTestsCount: integer;
    function GetTestName(Index: integer): string;
    function GetTestMethod(Index: integer): TSynTestEvent;
    function GetCount: Integer;
    function GetIdent: string;
  public
    /// create the test instance
    // - if an identifier is not supplied, the class name is used, after
    // T[Syn][Test] left trim and un-camel-case
    // - this constructor will add all published methods to the internal
    // test list, accessible via the Count/TestName/TestMethod properties
    constructor Create(const Ident: string = '');
    /// register a specified test to this class instance
    procedure Add(aMethod: TSynTestEvent; const aName: string);
    /// the test name
    // - either the Ident parameter supplied to the Create() method, either
    // a uncameled text from the class name
    property Ident: string read GetIdent;
    /// return the number of tests associated with this class
    // - i.e. the number of registered tests by the Register() method PLUS
    // the number of published methods defined within this class
    property Count: Integer read GetCount;
    /// get the name of a specified test
    // - Index range is from 0 to Count-1 (including)
    property TestName[Index: integer]: string read GetTestName;
    /// get the event of a specified test
    // - Index range is from 0 to Count-1 (including)
    property TestMethod[Index: integer]: TSynTestEvent read GetTestMethod;
    /// return the number of published methods defined within this class as tests
    // - i.e. the number of tests added by the Create() constructor from RTTI
    // - any TestName/TestMethod[] index higher or equal to this value has been
    // added by a specific call to the Add() method
    property InternalTestsCount: integer read fInternalTestsCount;
  published
    { all published methods of the children will be run as individual tests
      - these methods must be declared as procedure with no parameter }
  end;
{$M-}

  TSynTests = class;

  /// a class implementing a test case
  // - should handle a test unit, i.e. one or more tests
  // - individual tests are written in the published methods of this class
  TSynTestCase = class(TSynTest)
  protected
    fOwner: TSynTests;
    fAssertions: integer;
    fAssertionsFailed: integer;
    fAssertionsBeforeRun: integer;
    fAssertionsFailedBeforeRun: integer;
    fMethodIndex: integer;
    fTestCaseIndex: integer;
    /// any text assigned to this field will be displayed on console
    fRunConsole: string;
  public
    /// create the test case instance
    // - must supply a test suit owner
    // - if an identifier is not supplied, the class name is used, after
    // T[Syn][Test] left trim and un-camel-case
    constructor Create(Owner: TSynTests; const Ident: string = ''); virtual;
    /// used by the published methods to run a test assertion
    // - condition must equals TRUE to pass the test
    // - function return TRUE if the condition failed, in order to allow the
    // caller to stop testing with such code:
    // ! if Check(A=10) then exit;
    function Check(condition: Boolean; const msg: string = ''): Boolean;
      {$ifdef HASINLINE}inline;{$endif}
    /// used by the published methods to run a test assertion
    // - condition must equals FALSE to pass the test
    // - function return TRUE if the condition failed, in order to allow the
    // caller to stop testing with such code:
    // ! if CheckNot(A<>10) then exit;
    function CheckNot(condition: Boolean; const msg: string = ''): Boolean;
      {$ifdef HASINLINE}inline;{$endif}
    /// create a temporary string random content
    // - it somewhat faster if CharCount is a multiple of 5
    function RandomString(CharCount: Integer): RawByteString;
    /// create a temporary string random content
    // - it somewhat faster if CharCount is a multiple of 5
    function RandomUTF8(CharCount: Integer): RawUTF8;
    /// this method is trigerred internaly - e.g. by Check() - when a test failed
    procedure TestFailed(const msg: string);
    /// the test suit which owns this test case
    property Owner: TSynTests read fOwner;
    /// the test name
    // - either the Ident parameter supplied to the Create() method, either
    // an uncameled text from the class name
    property Ident: string read GetIdent;
    /// the number of assertions (i.e. Check() method call) for this test case
    property Assertions: integer read fAssertions;
    /// the number of assertions (i.e. Check() method call) for this test case
    property AssertionsFailed: integer read fAssertionsFailed;
    /// the index of the associated Owner.TestMethod[] which created this test
    property MethodIndex: integer read fMethodIndex;
    /// the index of the test case, starting at 0 for the associated MethodIndex
    property TestCaseIndex: integer read fTestCaseIndex;
  published
    { all published methods of the children will be run as individual tests
      - these methods must be declared as procedure with no parameter }
  end;

  TSynTestCaseClass = class of TSynTestCase;

  /// a class used to run a suit of test cases
  TSynTests = class(TSynTest)
  private
    function GetTestCase(Index: integer): TSynTestCase;
    function GetTestCaseCount: Integer;
    function GetFailedCaseIdent(Index: integer): string;
    function GetFailedCount: integer;
    function GetFailedMessage(Index: integer): string;
    function GetFailedCase(Index: integer): TSynTestCase;
  protected
    /// a list containing all failed tests after a call to the Run method
    // - if integer(Objects[]) is equal or higher than InternalTestsCount,
    // the Objects[] points to the TSynTestCase, and the Strings[] to the
    // associated failure message
    // - if integer(Objects[]) is lower than InternalTestsCount, then
    // it is an index to the corresponding published method, and the Strings[]
    // contains the associated failure message
    fFailed: TStringList;
    fTestCase: TObjectList;
    fAssertions: integer;
    fAssertionsFailed: integer;
    fCurrentMethod, fCurrentMethodFirstTestCaseIndex: integer;
    fSaveToFile: Text;
    /// this method is called during the run, for every testcase
    // - this implementation just report some minimal data to the console
    // by default, but may be overriden to update a real UI or reporting system
    // - the TestMethodIndex is first -1 before any TestMethod[] method call,
    // then called once after every TestMethod[] run
    procedure DuringRun(TestCaseIndex, TestMethodIndex: integer); virtual;
  public
    /// if set to a text file address, some debug messages will be reported to
    // this text file
    // - for example, use the following line to report to the console:
    // !  ToConsole := @Output;
    // - you can also use the SaveToFile() method to create an external file
    ToConsole: ^Text;
    /// you can put here some text to be displayed at the end of the messages
    // - some internal versions, e.g.
    // - every line of text must explicitly BEGIN with #13#10
    CustomVersions: string;
{$ifdef MSWINDOWS}
    /// contains the run elapsed time
    RunTimer: TPrecisionTimer;
{$endif}
    /// create the test instance
    // - if an identifier is not supplied, the class name is used, after
    // T[Syn][Test] left trim and un-camel-case
    // - this constructor will add all published methods to the internal
    // test list, accessible via the Count/TestName/TestMethod properties
    constructor Create(const Ident: string = '');
    /// finalize the class instance
    // - release all registered Test case instance
    destructor Destroy; override;
    /// save the debug messages into an external file
    // - if no file name is specified, the current Ident is used
    procedure SaveToFile(const DestPath: TFileName; const FileName: TFileName='');
    /// register a specified Test case instance
    // - all these instances will be freed by the TSynTests.Destroy
    // - the published methods of the children must call this method in order
    // to add test cases
    // - example of use (code from a TSynTests published method):
    // !  AddCase(TOneTestCase.Create(self));
    procedure AddCase(TestCase: TSynTestCase); overload;
      {$ifdef HASINLINE}inline;{$endif}
    /// register a specified Test case instance
    // - an instance of the supplied class is created, and will be freed by
    // TSynTests.Destroy
    // - the published methods of the children must call this method in order
    // to add test cases
    // - example of use (code from a TSynTests published method):
    // !  AddCase([TOneTestCase]);
    procedure AddCase(const TestCase: array of TSynTestCaseClass); overload;
    /// call of this method will run all associated tests cases
    // - function will return TRUE if all test passed
    // - all failed test cases will be added to the Failed[] list
    // - the TestCase[] list is created first, by running all published methods,
    // which must call the AddCase() method above to register test cases
    // - the Failed[] list is cleared at the beginning of the run
    // - Assertions and AssertionsFailed properties are reset and computed during
    // the run
    function Run: Boolean; virtual;
    /// the number of items in the TestCase[] array
    property TestCaseCount: Integer read GetTestCaseCount;
    /// an array containing all registered Test case instances
    // - Test cases are registered by the AddCase() method above, mainly
    // by published methods of the children
    // - Test cases instances are freed by TSynTests.Destroy
    property TestCase[Index: integer]: TSynTestCase read GetTestCase;
    /// number of failed tests after the last call to the Run method
    property FailedCount: integer read GetFailedCount;
    /// retrieve the TSynTestCase instance associated with this failure
    // - returns nil if this failure was not trigerred by a TSynTestCase,
    // but directly by a method
    property FailedCase[Index: integer]: TSynTestCase read GetFailedCase;
    /// retrieve the ident of the case test associated with this failure
    property FailedCaseIdent[Index: integer]: string read GetFailedCaseIdent;
    /// retrieve the error message associated with this failure
    property FailedMessage[Index: integer]: string read GetFailedMessage;
    /// the number of assertions (i.e. Check() method call) in all tests
    // - this property is set by the Run method above
    property Assertions: integer read fAssertions;
    /// the number of assertions (i.e. Check() method call) which failed in all tests
    // - this property is set by the Run method above
    property AssertionsFailed: integer read fAssertionsFailed;
  published
    { all published methods of the children will be run as test cases registering
      - these methods must be declared as procedure with no parameter
      - every method should create a customized TSynTestCase instance,
        which will be registered with the AddCase() method, then automaticaly
        destroyed during the TSynTests destroy  }
  end;

  /// this test case will test most functions, classes and types defined and
  // implemented in the SynCommons unit
  TTestLowLevelCommon = class(TSynTestCase)
  published
    /// the faster CopyRecord function, enhancing the system.pas unit
    procedure SystemCopyRecord;
    /// test StrIComp() and AnsiIComp() functions
    procedure FastStringCompare;
    /// test UrlEncode() and UrlDecode() functions
    procedure UrlEncoding;
    /// the Soundex search feature (i.e. TSynSoundex and all related
    // functions)
    procedure Soundex;
    /// low level fast Integer or Floating-Point to/from string conversion
    // - especially the RawUTF8 or PUTF8Char relative versions
    procedure NumericalConversions;
    /// the new fast Currency to/from string conversion
    procedure Curr64;
    /// the camel-case / camel-uncase features, used for i18n from Delphi RTII
    procedure _CamelCase;
    /// the low-level bit management functions
    procedure Bits;
    /// the fast .ini file content direct access
    procedure IniFiles;
    /// UTF-8 and Win-Ansi conversion (from or to, through RawUnicode)
    procedure _UTF8;
  end;

/// convert a size to a human readable value
// - append MB, KB or B symbol
// - for MB and KB, add one fractional digit
function KB(bytes: cardinal): RawUTF8;

/// convert a micro seconds elapsed time into a human readable value
// - append us, ms or s symbol
// - for us and ms, add two fractional digits
function MicroSecToString(Micro: Int64): RawUTF8;

/// return the Delphi Compiler Version
// - returns 'Delphi 2007' or 'Delphi 2010' e.g.
function GetDelphiCompilerVersion: RawUTF8;


implementation

{ ************ some fast UTF-8 / Unicode / Ansi conversion routines }

function UnicodeCharToUtf8(Dest: PUTF8Char; aWideChar: PtrUInt): integer;
begin
  if aWideChar<=$7F then begin
    Dest^ := AnsiChar(aWideChar);
    result := 1;
  end else
  if aWideChar>$7ff then begin
    Dest[0] := AnsiChar($E0 or (aWideChar shr 12));
    Dest[1] := AnsiChar($80 or ((aWideChar shr 6) and $3F));
    Dest[2] := AnsiChar($80 or (aWideChar and $3F));
    result := 3;
  end else begin
    Dest[0] := AnsiChar($C0 or (aWideChar shr 6));
    Dest[1] := AnsiChar($80 or (aWideChar and $3F));
    result := 2;
  end;
end;

function WinAnsiBufferToUtf8(Dest: PUTF8Char; Source: PAnsiChar; SourceChars: Cardinal): PUTF8Char;
// length(Dest) must be reserved with at least SourceChars*3
// a #0 char is appened at the end
var EndSource: PAnsiChar;
    c: Cardinal;
begin
  if Dest=nil then begin
    Result := nil;
    Exit;
  end else
  if (Source<>nil) and (SourceChars>0) then begin
    EndSource := Source+SourceChars;
    repeat
      c := byte(source^); inc(source);
      if c <= $7F then begin
        Dest^ := AnsiChar(c); // 0..127 don't need any translation
        Inc(Dest);
        if source<endsource then continue else break;
      end
      else begin
        c := WinAnsiTable[c]; // convert WinAnsi char into Unicode char
        if c > $7ff then begin
          Dest[0] := AnsiChar($E0 or (c shr 12));
          Dest[1] := AnsiChar($80 or ((c shr 6) and $3F));
          Dest[2] := AnsiChar($80 or (c and $3F));
          Inc(Dest,3);
          if source<endsource then continue else break;
        end else begin
          Dest[0] := AnsiChar($C0 or (c shr 6));
          Dest[1] := AnsiChar($80 or (c and $3F));
          Inc(Dest,2);
          if source<endsource then continue else break;
        end;
      end;
    until false;
  end;
  Dest^ := #0;
  Result := Dest;
end;

var
  WinAnsiTableA: packed array[AnsiChar] of Word absolute WinAnsiTable;

procedure WinAnsiToUnicodeBuffer(const S: WinAnsiString; Dest: PWordArray; DestLen: integer);
var i, L: PtrInt;
begin
  L := PtrInt(S);
  if L<>0 then begin
    L := PPtrInt(L-4)^;
    if L>=DestLen then
      L := DestLen-1;
    for i := 0 to L do // include S[L+1] = last #0
      Dest^[i] := WinAnsiTableA[S[i+1]]; // very fast conversion
  end else
    Dest^[0] := 0;
end;

function WinAnsiToRawUnicode(const S: WinAnsiString): RawUnicode;
var i, L: PtrInt;
    PW: PWordArray;
begin
  result := '';
  if S='' then
    exit;
  L := PPtrInt(PtrInt(S)-4)^;
  SetLength(result,L*2+1); // +1 to include last WideChar(#0)
{$ifdef UNICODE2} // not needed: SetLength() did already set the codepage
  PWord(PtrInt(result)-12)^ := 1200; // Codepage for an UnicodeString
{$endif}
  PW := pointer(result);
  for i := 0 to L do // include S[L+1] = last #0
    PW^[i] := WinAnsiTableA[S[i+1]]; // very fast conversion
end;

function WinAnsiToUtf8(const S: WinAnsiString): RawUTF8;
// faster than SysUtils: don't use Utf8Encode(WideString) -> no Windows.Global()
var L: integer;
    tmp: RawByteString; // don't use result as temporary buffer
begin
  result := '';
  if S='' then
    exit;
  L := PPtrInt(PtrInt(S)-4)^;
  SetLength(tmp,L*3); // enough place
  SetString(result,PAnsiChar(pointer(tmp)),
    WinAnsiBufferToUtf8(pointer(tmp),pointer(S),L)-pointer(tmp));
end;

function WideCharToWinAnsiChar(wc: cardinal): AnsiChar;
begin // code generated for this function is very fast
  if wc<256 then
    if WinAnsiTable[wc]<256 then begin
      result := AnsiChar(wc);
      exit;
    end else begin
      result := ' '; // invalid ansi char for this code page (e.g. #128)
      exit;
    end else begin // wc>255:
//    FastFindIntegerSorted(@WinAnsiTableSortedWide,31,wc); 
    for result := #128 to #159 do // chars >#255 are only in this range
      if WinAnsiUnicodeChars[ord(result)]=wc then
        exit;
    result := ' '; // space for invalid wide char
    exit;
  end;
end;

function WinAnsiTableSortedFind(wc: cardinal): Integer;
var L,R: PtrInt;
    cmp: integer;
begin
    L := 0;
    R := high(WinAnsiTableSortedWide);
    repeat // very fast binary search
      result := (L + R) shr 1;
      cmp := WinAnsiTableSortedWide[result]-integer(wc);
      if cmp=0 then begin
        result := WinAnsiTableSortedAnsi[result];
        exit;
      end;
      if cmp<0 then
        L := result + 1 else
        R := result - 1;
    until (L > R);
  result := -1; // invalid wide char
end;

function WideCharToWinAnsi(wc: cardinal): integer;
begin
  if wc<256 then
    if WinAnsiTable[wc]<256 then
      result := wc else
      result := -1 else // invalid ansi char for this code page (e.g. #128)
      result := WinAnsiTableSortedFind(wc);
end;

function IsWinAnsi(WideText: PWideChar; Length: integer): boolean;
var i: integer;
begin
  result := false;
  if WideText<>nil then
    for i := 1 to Length do
      if WideCharToWinAnsi(cardinal(WideText^))<0 then
        exit else
        inc(WideText);
  result := true;
end;

function IsAnsiCompatible(PC: PAnsiChar): boolean;
begin
  result := false;
  if PC<>nil then
  while true do
    if PC^=#0 then
      break else
    if PC^<=#127 then
      inc(PC) else // 7 bits chars are always OK, whatever codepage/charset is used
      exit;
  result := true;
end;

function IsWinAnsi(WideText: PWideChar): boolean;
var L,R,pivot: PtrInt;
    cmp: integer;
    wc: integer;
begin
  result := false;
  if WideText<>nil then
    repeat
      wc := cardinal(WideText^);
      inc(WideText);
      if wc=0 then
        Break else
      if wc<256 then
        if WinAnsiTable[wc]<256 then
          continue else
          exit else begin
          // wc>=256 -> fast binary search of the Unicode value in WinAnsiTable
          L := 0;
          R := high(WinAnsiTableSortedWide);
          repeat // very fast binary search
            pivot := (L + R) shr 1;
            cmp := WinAnsiTableSortedWide[pivot]-wc;
            if cmp=0 then
              break else // found
            if cmp<0 then
              L := pivot + 1 else
              R := pivot - 1;
          until (L > R);
          if cmp<>0 then
            exit; // invalid wide char
        end;
    until false;
  result := true;
end;

function IsWinAnsiU(UTF8Text: PUTF8Char): boolean;
var c: Cardinal;
begin
  result := false;
  if UTF8Text<>nil then
    repeat
      c := byte(UTF8Text^); inc(UTF8Text);
      if c=0 then break else
      if c and $80=0 then begin
        continue;
      end else begin
        if UTF8Text^=#0 then break;
        if c and $20=0 then begin
          c := c shl 6+byte(UTF8Text^)-$00003080; inc(UTF8Text);
        end else begin
          c := c shl 6+byte(UTF8Text^); inc(UTF8Text);
          if UTF8Text^=#0 then break;
          c := c shl 6+byte(UTF8Text^)-$000E2080; inc(UTF8Text);
        end;
        if c>255 then begin
          if WinAnsiTableSortedFind(c)<0 then
            exit; // invalid char in the WinAnsi code page
        end else
        if WinAnsiTable[c]>255 then
          exit; // invalid char in the WinAnsi code page
      end;
    until false;
  result := true;
end;

function UTF8ToWinPChar(dest: PAnsiChar; source: PUTF8Char; count: integer): integer;
var c: cardinal;
    begd: PAnsiChar;
    endSource: PUTF8Char;
begin
  result := 0;
  if source=nil then exit;
  begd := dest;
  endSource := source+count;
  repeat
    c := byte(source^); inc(source);
    if byte(c) and $80=0 then begin
      dest^ := AnsiChar(byte(c)); inc(dest);
      if source<endsource then continue else break;
    end else begin
      if source>=endsource then break;
      if c and $20=0 then begin
        c := c shl 6+byte(source^)-$00003080; inc(source);
        if c and $ffffff00=0 then begin
          if WinAnsiTable[c]>255 then
            dest^ := ' ' else // invalid char in the WinAnsi code page
            dest^ := AnsiChar(c);
          inc(dest);  // #128..#255 -> direct copy
          if source<endsource then continue else break;
        end;
      end else begin
        c := c shl 6+byte(source^); inc(source);
        if source>=endsource then break;
        c := c shl 6+byte(source^)-$000E2080; inc(source);
      end;
      // #256.. -> slower but accurate conversion
      dest^ := WideCharToWinAnsiChar(c); 
      if dest^<>' ' then inc(dest);
      if source>=endsource then break;
    end;
  until false;
  result := dest-begd;
end;

procedure UTF8ToShortString(var dest: shortstring; source: PUTF8Char);
var c: cardinal;
    len: integer;
begin
  len := 0;
  if source<>nil then
  repeat
    c := byte(source^); inc(source);
    if c=0 then break else
    if c and $80=0 then begin
      inc(len); dest[len] := AnsiChar(c);
      if len<255 then continue else break;
    end else begin
      if source^=#0 then break;
      if c and $20=0 then begin
        c := c shl 6+byte(source^)-$00003080; inc(source);
      end else begin
        c := c shl 6+byte(source^); inc(source);
        if source^=#0 then break;
        c := c shl 6+byte(source^)-$000E2080; inc(source);
      end;
      inc(len);
      if c>255 then
        dest[len] := WideCharToWinAnsiChar(c) else
        if WinAnsiTable[c]>255 then
          dest[len] := '?' else // invalid char in the WinAnsi code page
          dest[len] := AnsiChar(byte(c)); // #128..#255 -> direct copy
      if len<255 then continue else break;
    end;
  until false;
  dest[0] := AnsiChar(len);
end;

function Utf8ToWinAnsi(const S: RawUTF8): WinAnsiString;
var L: integer;
    tmp: RawByteString; // don't use result as temporary buffer
begin
  result := '';
  if S='' then
    exit;
  L := PPtrInt(PtrInt(S)-4)^;
  SetLength(tmp,L);
  SetString(result,PAnsiChar(pointer(tmp)),
    UTF8ToWinPChar(pointer(tmp),pointer(S),L));
end;

function Utf8ToWinAnsi(P: PUTF8Char): WinAnsiString;
var L: integer;
    tmp: RawByteString; // don't use result as temporary buffer
begin
  result := '';
  L := StrLen(P);
  if L=0 then
    exit;
  SetLength(tmp,L);
  SetString(result,PAnsiChar(pointer(tmp)),UTF8ToWinPChar(pointer(tmp),P,L));
end;

procedure Utf8ToRawUTF8(P: PUTF8Char; var result: RawUTF8);
begin // fast and Delphi 2009/2010 ready
  SetString(result,PAnsiChar(P),StrLen(P));
end;

function UTF8ToWideChar(dest: pWideChar; source: PUTF8Char; sourceBytes: integer=0): PtrInt;
// faster than System.UTF8Decode()
var c: cardinal;
    begd: pWideChar;
    endSource: PUTF8Char;
begin
  result := 0;
  if source=nil then
   exit;
  if sourceBytes=0 then begin
    if source^=#0 then
      exit;
    endSource := source;
    while endsource^<>#0 do
      inc(endsource);
  end else
    endSource := source+sourceBytes;
  begd := dest;
  repeat
    c := byte(source^);
    inc(source);
    if c and $80=0 then begin
      dest^ := WideChar(c);
      inc(dest);
      if source<endsource then
        continue else
        break;
    end else
      if source>=endsource then
        break;
      if c and $20=0 then begin
        c := c shl 6+byte(source^)-$00003080;
        inc(source);
        dest^ := WideChar(c);
        inc(dest);
        if source>=endsource then
          break;
      end else begin
        c := c shl 6+byte(source^);
        inc(source);
        if source>=endsource then
          break;
        c := c shl 6+byte(source^)-$000E2080;
        inc(source);
        dest^ := WideChar(c);
        inc(dest);
        if source>=endsource then
          break;
      end;
  until false;
  dest^ := #0; // always append a WideChar(0) to the end of the buffer
  result := PtrInt(dest)-PtrInt(begd); // dest-begd return char length
end;

function Utf8ToUnicodeLength(source: PUTF8Char): PtrInt;
var c: byte;
begin
  result := 0;
  if source<>nil then
  repeat
    c := byte(source^); inc(source);
    if c=0 then break else
    if c and $80=0 then
      inc(result) else begin
      if source^=#0 then break else inc(source);
      if c and $20<>0 then
        if source^=#0 then break else inc(source);
      inc(result);
    end;
  until false;
end;

function Utf8FirstLineToUnicodeLength(source: PUTF8Char): PtrInt;
var c: byte;
begin
  result := 0;
  if source<>nil then
  repeat
    c := byte(source^); inc(source);
    if c in [0,10,13] then break else // #0, #10 or #13 stop the count
    if c and $80=0 then
      inc(result) else begin
      if source^=#0 then break else inc(source);
      if c and $20<>0 then
        if source^=#0 then break else inc(source);
      inc(result);
    end;
  until false;
end;

function Utf8DecodeToRawUnicode(P: PUTF8Char; L: integer): RawUnicode; overload;
var tmp: RawByteString;
begin
  result := ''; // somewhat faster if result is freed before any SetLength()
  if L=0 then
    L := StrLen(P);
  if L=0 then
    exit;
  // +1 below is for #0 ending -> true WideChar(#0) ending
  SetLength(tmp,L*2+1); // maximum posible unicode size (if all <#128)
  SetString(result,PAnsiChar(pointer(tmp)),
    UTF8ToWideChar(pointer(tmp),P,L)+1);
end;

function Utf8DecodeToRawUnicode(const S: RawUTF8): RawUnicode; overload;
begin
  if S='' then
    result := '' else
    result := Utf8DecodeToRawUnicode(pointer(S),PInteger(PtrInt(S)-4)^);
end;

function Utf8DecodeToRawUnicodeUI(const S: RawUTF8): RawUnicode; overload;
var L: integer;
begin
  result := ''; // somewhat faster if result is freed before any SetLength()
  if S='' then
    exit;
  L := PPtrInt(PtrInt(S)-4)^;
  SetLength(result,L*2+1);
{$ifdef UNICODE2} // not needed: SetLength() did already set the codepage
  PWord(PtrInt(result)-12)^ := 1200; // Codepage for an UnicodeString
{$endif}
  UTF8ToWideChar(pointer(result),Pointer(S),L);
end;

function RawUnicodeToUtf8(Dest: PUTF8Char; DestLen: PtrInt; Source: PWideChar; SourceLen: PtrInt): PtrInt; overload;
var c: PtrUInt;
begin
  result := PtrInt(Dest);
  if (Source<>nil) and (Dest<>nil) then begin
    Inc(DestLen,PtrInt(Dest));
    SourceLen := SourceLen*2+PtrInt(Source);
    if (PtrInt(Dest)<DestLen) and (PtrInt(Source)<SourceLen) then
    repeat
      c := PtrUInt(Source^);
      inc(Source);
      if c<=$7F then begin
        Dest^ := AnsiChar(c);
        inc(Dest);
        if (PtrInt(Dest)<DestLen) and (PtrInt(Source)<SourceLen) then continue else break;
      end else
      if c>$7ff then begin
        Dest[0] := AnsiChar($E0 or (c shr 12));
        Dest[1] := AnsiChar($80 or ((c shr 6) and $3F));
        Dest[2] := AnsiChar($80 or (c and $3F));
        inc(Dest,3);
        if (PtrInt(Dest)<DestLen) and (PtrInt(Source)<SourceLen) then continue else break;
      end else begin
        Dest[0] := AnsiChar($C0 or (c shr 6));
        Dest[1] := AnsiChar($80 or (c and $3F));
        inc(Dest,2);
        if (PtrInt(Dest)<DestLen) and (PtrInt(Source)<SourceLen) then continue else break;
      end;
    until false;
    Dest^ := #0;
  end;
  result := PtrInt(Dest)-result;
end;

/// convert a RawUnicode PChar into a UTF-8 string
function RawUnicodeToUtf8(P: PWideChar; WideCharCount: integer): RawUTF8;
var L,LW: integer;
begin
  result := ''; // somewhat faster if result is freed before any SetLength()
  if WideCharCount=0 then
    exit;
  LW := WideCharCount*3; // maximum resulting length
  SetLength(result,LW);
{$ifdef UNICODE2} // not needed: SetLength() did already set the codepage
  PWord(PtrInt(result)-12)^ := CP_UTF8; // use only SetLength() -> force set code page
{$endif}
  L := RawUnicodeToUtf8(pointer(result),LW+1,P,WideCharCount);
  if L<=0 then
    result := '' else
    if L<>LW then
      SetLength(result,L);
end;

function RawUnicodeToUtf8(P: PWideChar; WideCharCount: integer; out UTF8Length: integer): RawUTF8; overload;
var LW: integer;
begin
  result := ''; // somewhat faster if result is freed before any SetLength()
  if WideCharCount=0 then
    exit;
  LW := WideCharCount*3; // maximum resulting length
  SetLength(result,LW);
{$ifdef UNICODE2} // not needed: SetLength() did already set the codepage
  PWord(PtrInt(result)-12)^ := CP_UTF8; // use only SetLength() -> force set code page
{$endif}
  UTF8Length := RawUnicodeToUtf8(pointer(result),LW+1,P,WideCharCount);
  if UTF8Length<=0 then
    result := '';
end;

/// convert a RawUnicode string into a UTF-8 string
function RawUnicodeToUtf8(const Unicode: RawUnicode): RawUTF8;
begin
  result := RawUnicodeToUtf8(pointer(Unicode),length(Unicode) shr 1);
end;

procedure RawUnicodeToWinPChar(dest: PAnsiChar; source: PWideChar; WideCharCount: integer);
begin
  while WideCharCount>0 do begin
    dest^ := WideCharToWinAnsiChar(cardinal(source^)); // UNICODE -> fast inline
    inc(source);
    inc(dest);
    dec(WideCharCount);
  end;
end;

function RawUnicodeToWinAnsi(P: PWideChar; WideCharCount: integer): WinAnsiString; overload;
begin
  result := ''; // somewhat faster if result is freed before any SetLength()
  if WideCharCount=0 then
    exit;
  SetLength(result,WideCharCount);
{$ifdef UNICODE2} // not needed: SetLength() did already set the codepage
  PWord(PtrInt(result)-12)^ := 1252; // use only SetLength() -> force set code page
{$endif}
  RawUnicodeToWinPChar(pointer(result),P,WideCharCount);
end;

function RawUnicodeToWinAnsi(const Unicode: RawUnicode): WinAnsiString; overload;
begin
  result := RawUnicodeToWinAnsi(pointer(Unicode),length(Unicode) shr 1);
end;

procedure UnicodeBufferToWinAnsi(source: PWideChar; out Dest: WinAnsiString);
var L: integer;
begin
  L := StrLenW(source);
  SetLength(Dest,L);
  RawUnicodeToWinPChar(pointer(Dest),source,L);
end;

{$ifdef UNICODE}

/// convert a Delphi 2009 Unicode string into our UTF-8 string
function UnicodeStringToUtf8(const S: string): RawUTF8;
begin
  result := RawUnicodeToUtf8(pointer(S),length(S));
end;

function UnicodeStringToWinAnsi(const S: string): WinAnsiString;
begin
  result := RawUnicodeToWinAnsi(pointer(S),length(S));
end;

function UTF8DecodeToUnicodeString(const S: RawUTF8): UnicodeString; overload;
begin
  result := UTF8DecodeToUnicodeString(pointer(S),length(S));
end;

function UTF8DecodeToUnicodeString(P: PUTF8Char; L: integer): UnicodeString; overload;
var L2: integer;
begin
  result := ''; // somewhat faster if result is freed before any SetLength()
  if L=0 then
    L := StrLen(P);
  if L=0 then
    exit;
  SetLength(result,L); // maximum posible unicode size (if all <#128)
  L2 := UTF8ToWideChar(pointer(result),P,L) shr 1;
  if L2<>L then
    SetLength(result,L2);
end;

{$endif}

function StringToUTF8(const Text: string): RawUTF8;
{$ifdef UNICODE}
begin
  result := RawUnicodeToUtf8(pointer(Text),length(Text));
end;
{$else}
var Dest: RawUnicode;
    L: integer;
begin
  result := '';
  L := length(Text);
  if L=0 then
    exit;
  SetLength(Dest,L*2+1);
  StringToWideChar(Text,pointer(Dest),L*2+1); // low-level Delphi RTL
  result := RawUnicodeToUtf8(pointer(Dest),StrLenW(pointer(Dest)));
end;
{$endif}

function UTF8ToString(const Text: RawUTF8): string;
{$ifdef UNICODE}
begin
  result := UTF8DecodeToUnicodeString(pointer(Text),length(Text));
end;
{$else}
var Dest: RawUnicode;
    L: integer;
begin
  result := '';
  L := length(Text);
  if L=0 then
    exit;
  SetLength(Dest,L*2);
  L := UTF8ToWideChar(pointer(Dest),pointer(Text),L) shr 1;
  WideCharLenToStrVar(pointer(Dest),L,result); // low-level Delphi RTL
end;
{$endif}

function StrInt32(P: PAnsiChar; val: PtrInt): PAnsiChar;
var c,c10: PtrUInt;
begin // this code is faster than the Borland's original str() or IntToStr()
  if val<0 then
    c := -val else
    c := val;
  repeat
    if c<10 then begin
      dec(P);
      P^ := AnsiChar(c+ord('0'));
      break;
    end;
    c10 := c div 100;   // one div by two digits
{$ifdef CPU64} // CPU64 has no LEA optimization for *10 but has one cycle mul
    dec(c,c10*100);     {$else}
    dec(c,(c10*10)*10); // fast c := c mod 100
{$endif}
    dec(P,2);
    PWord(P)^ := TwoDigitLookupW[c];
    c := c10;
    if c10=0 then break;
  until false;
  if val<0 then begin
    dec(P);
    P^ := '-';
  end;
  result := P;
end;

{$ifndef CPU64} // StrInt32 aldready implemented PtrInt=Int64
function StrInt64(P: PAnsiChar; val: Int64): PAnsiChar;
var c,c10: Int64;
begin // this code is faster than the Borland's original str() or IntToStr()
  if val<0 then
    c := -val else
    c := val;
  repeat
    if c<maxInt then begin
      P := StrInt32(P,Int64Rec(c).Lo);
      break;
    end;
    c10 := c div 100;   // one div by two digits
    dec(c,c10*100);     // fast c := c mod 100
    dec(P,2);
    PWord(P)^ := TwoDigitLookupW[c];
    c := c10;
    if c10=0 then break;
  until false;
  if val<0 then begin
    dec(P);
    P^ := '-';
  end;
  result := P;
end;
{$endif}



type
  PStrRec = ^TStrRec;
  /// map the Delphi string header, as defined in System.pas
  TStrRec = packed record
{$ifdef UNICODE}
    /// the associated code page used for this string
    // - exist only since Delphi 2009
    // - 0 or 65535 for RawByteString
    // - 1200=CP_UTF16 for UnicodeString
    // - 65001=CP_UTF8 for RawUTF8
    // - the current code page for AnsiString
    codePage: Word;
    /// either 1 (for AnsiString) or 2 (for UnicodeString)
    // - exist only since Delphi 2009
    elemSize: Word;
{$endif}
    /// string reference count (basic garbage memory mechanism)
    refCnt: Longint;
    /// length in characters
    // - size in bytes = length*elemSize
    length: Longint;
  end;

const
  /// codePage offset = string header size
  // - used to calc the beginning of memory allocation of a string 
  STRRECSIZE = SizeOf(TStrRec);

  
{$ifndef ENHANCEDRTL} { our Enhanced Runtime (or LVCL) library contain fast asm versions }

{ code below was extracted from our Enhanced Runtime (or LVCL) library
   and increases the framework performance
  - not compiled with FPC, since does call some low-level system.pas functions  }

{$ifndef PUREPASCAL} { these functions are implemented in asm }
{$ifndef LVCL} { don't define these functions twice }
{$ifndef FPC}  { these asm function use some low-level system.pas calls }

{$define OWNI2S}

function Int32ToUTF8(Value : integer): RawUTF8; // 3x faster than SysUtils.IntToStr
// from IntToStr32_JOH_IA32_6_a, adapted for Delphi 2009/2010
asm
  push   ebx
  push   edi
  push   esi
  mov    ebx,eax                {Value}
  sar    ebx,31                 {0 for +ve Value or -1 for -ve Value}
  xor    eax,ebx
  sub    eax,ebx                {ABS(Value)}
  mov    esi,10                 {Max Digits in result}
  mov    edi,edx                {@result}
  cmp    eax,10;         sbb    esi, 0
  cmp    eax,100;        sbb    esi, 0
  cmp    eax,1000;       sbb    esi, 0
  cmp    eax,10000;      sbb    esi, 0
  cmp    eax,100000;     sbb    esi, 0
  cmp    eax,1000000;    sbb    esi, 0
  cmp    eax,10000000;   sbb    esi, 0
  cmp    eax,100000000;  sbb    esi, 0
  cmp    eax,1000000000; sbb    esi, ebx    {esi=Digits (Including Sign Character)}
  mov    ecx,[edx]              {result}
  test   ecx,ecx
  je     @@NewStr               {Create New string for result}
  cmp    dword ptr [ecx-8], 1
  jne    @@ChangeStr            {Reference Count <> 1}
  cmp    esi,[ecx-4]
  je     @@LengthOk             {Existing Length = Required Length}
  sub    ecx,STRRECSIZE         {Allocation Address}
  push   eax                    {ABS(Value)}
  push   ecx
  mov    eax,esp
  lea    edx,[esi+STRRECSIZE+1] {New Allocation Size}
  call   system.@ReallocMem     {Reallocate result string}
  pop    ecx
  pop    eax                    {ABS(Value)}
  add    ecx,STRRECSIZE         {result}
  mov    [ecx-4],esi            {Set New Length}
  mov    byte ptr [ecx+esi],0   {Add Null Terminator}
  mov    [edi],ecx              {Set result Address}
  jmp    @@LengthOk
@@ChangeStr:
  mov     edx,dword ptr [ecx-8]  {Reference Count}
  add     edx,1
  jz      @@NewStr               {RefCount = -1 (string Constant)}
  lock    dec dword ptr [ecx-8]  {Decrement Existing Reference Count}
@@NewStr:
  push   eax                     {ABS(Value)}
  mov    eax,esi                 {Length}
{$ifdef UNICODE}
  mov    edx,CP_UTF8 // UTF-8 code page for Delphi 2009/2010
{$endif}
  call   system.@NewAnsiString
  mov    [edi],eax               {Set result Address}
  mov    ecx,eax                 {result}
  pop    eax                     {ABS(Value)}
@@LengthOk:
  mov    byte ptr [ecx],'-'      {Store '-' Character (May be Overwritten)}
  add    esi,ebx                 {Digits (Excluding Sign Character)}
  sub    ecx,ebx                 {Destination of 1st Digit}
  sub    esi,2                   {Digits (Excluding Sign Character) - 2}
  jle    @@FinalDigits           {1 or 2 Digit Value}
  cmp    esi,8                   {10 Digit Value?}
  jne    @@SetResult             {Not a 10 Digit Value}
  sub    eax,2000000000          {Digit 10 must be either '1' or '2'}
  mov    dl,'2'
  jnc    @@SetDigit10            {Digit 10 = '2'}
  mov    dl,'1'                  {Digit 10 = '1'}
  add    eax,1000000000
@@SetDigit10:
  mov    [ecx],dl                {Save Digit 10}
  mov    esi,7                   {9 Digits Remaining}
  add    ecx,1                   {Destination of 2nd Digit}
@@SetResult:
  mov    edi,$28F5C29            {((2^32)+100-1)/100}
@@Loop:
  mov    ebx,eax                 {Dividend}
  mul    edi                     {EDX = Dividend DIV 100}
  mov    eax,edx                 {Set Next Dividend}
  imul   edx,-200                {-2 * (100 * Dividend DIV  100)}
  movzx  edx,word ptr [TwoDigitLookup+ebx*2+edx] {Dividend MOD 100 in ASCII}
  mov    [ecx+esi],dx
  sub    esi,2
  jg     @@Loop                  {Loop until 1 or 2 Digits Remaining}
@@FinalDigits:
  pop    esi
  pop    edi
  pop    ebx
  jnz    @@LastDigit
  movzx  eax,word ptr [TwoDigitLookup+eax*2]
  mov    [ecx],ax                {Save Final 2 Digits}
  ret
@@LastDigit:
  or     al,'0'                  {Ascii Adjustment}
  mov    [ecx],al                {Save Final Digit}
end;

function Int64ToUTF8(Value: Int64): RawUTF8;
// from IntToStr64_JOH_IA32_6_b, adapted for Delphi 2009/2010 
asm
  push   ebx
  mov    ecx, [ebp+8]            {Low Integer of Value}
  mov    edx, [ebp+12]           {High Integer of Value}
  xor    ebp, ebp                {Clear Sign Flag (EBP Already Pushed)}
  mov    ebx, ecx                {Low Integer of Value}
  test   edx, edx
  jnl    @@AbsValue
  mov    ebp, 1                  {EBP = 1 for -ve Value or 0 for +ve Value}
  neg    ecx
  adc    edx, 0
  neg    edx
@@AbsValue:                      {EDX:ECX = Abs(Value)}
  jnz    @@Large
  test   ecx, ecx
  js     @@Large
  mov    edx, eax                {@Result}
  mov    eax, ebx                {Low Integer of Value}
  call   Int32ToUTF8               {Call Fastest Integer IntToStr Function}
  pop    ebx
@@Exit:
  pop    ebp                     {Restore Stack and Exit}
  ret    8
@@Large:
  push   edi
  push   esi
  mov    edi, eax
  xor    ebx, ebx
  xor    eax, eax
@@Test15:                        {Test for 15 or More Digits}
  cmp    edx, $00005af3          {100000000000000 div $100000000}
  jne    @@Check15
  cmp    ecx, $107a4000          {100000000000000 mod $100000000}
@@Check15:
  jb     @@Test13
@@Test17:                        {Test for 17 or More Digits}
  cmp    edx, $002386f2          {10000000000000000 div $100000000}
  jne    @@Check17
  cmp    ecx, $6fc10000          {10000000000000000 mod $100000000}
@@Check17:
  jb     @@Test15or16
@@Test19:                        {Test for 19 Digits}
  cmp    edx, $0de0b6b3          {1000000000000000000 div $100000000}
  jne    @@Check19
  cmp    ecx, $a7640000          {1000000000000000000 mod $100000000}
@@Check19:
  jb     @@Test17or18
  mov    al, 19
  jmp    @@SetLength
@@Test17or18:                    {17 or 18 Digits}
  mov    bl, 18
  cmp    edx, $01634578          {100000000000000000 div $100000000}
  jne    @@SetLen
  cmp    ecx, $5d8a0000          {100000000000000000 mod $100000000}
  jmp    @@SetLen
@@Test15or16:                    {15 or 16 Digits}
  mov    bl, 16
  cmp    edx, $00038d7e          {1000000000000000 div $100000000}
  jne    @@SetLen
  cmp    ecx, $a4c68000          {1000000000000000 mod $100000000}
  jmp    @@SetLen
@@Test13:                        {Test for 13 or More Digits}
  cmp    edx, $000000e8          {1000000000000 div $100000000}
  jne    @@Check13
  cmp    ecx, $d4a51000          {1000000000000 mod $100000000}
@@Check13:
  jb     @@Test11
@@Test13or14:                    {13 or 14 Digits}
  mov    bl, 14
  cmp    edx, $00000918          {10000000000000 div $100000000}
  jne    @@SetLen
  cmp    ecx, $4e72a000          {10000000000000 mod $100000000}
  jmp    @@SetLen
@@Test11:                        {10, 11 or 12 Digits}
  cmp    edx, $02                {10000000000 div $100000000}
  jne    @@Check11
  cmp    ecx, $540be400          {10000000000 mod $100000000}
@@Check11:
  mov    bl, 11
  jb     @@SetLen                {10 Digits}
@@Test11or12:                    {11 or 12 Digits}
  mov    bl, 12
  cmp    edx, $17                {100000000000 div $100000000}
  jne    @@SetLen
  cmp    ecx, $4876e800          {100000000000 mod $100000000}
@@SetLen:
  sbb    eax, 0                  {Adjust for Odd/Evem Digit Count}
  add    eax, ebx
@@SetLength:                     {Abs(Value) in EDX:ECX, Digits in EAX}
  push   ecx                     {Save Abs(Value)}
  push   edx
  lea    edx, [eax+ebp]          {Digits Needed (Including Sign Character)}
  mov    ecx, [edi]              {@Result}
  mov    esi, edx                {Digits Needed (Including Sign Character)}
  test   ecx, ecx
  je     @@NewStr                {Create New AnsiString for Result}
  cmp    dword ptr [ecx-8], 1
  jne    @@ChangeStr             {Reference Count <> 1}
  cmp    esi, [ecx-4]
  je     @@LengthOk              {Existing Length = Required Length}
  sub    ecx, STRRECSIZE         {Allocation Address}
  push   eax                     {ABS(Value)}
  push   ecx
  mov    eax, esp
  lea    edx, [esi+STRRECSIZE+1] {New Allocation Size}
  call   system.@ReallocMem      {Reallocate Result AnsiString}
  pop    ecx
  pop    eax                     {ABS(Value)}
  add    ecx, STRRECSIZE         {@Result}
  mov    [ecx-4], esi            {Set New Length}
  mov    byte ptr [ecx+esi], 0   {Add Null Terminator}
  mov    [edi], ecx              {Set Result Address}
  jmp    @@LengthOk
@@ChangeStr:
  mov     edx, dword ptr [ecx-8]  {Reference Count}
  add     edx, 1
  jz      @@NewStr                {RefCount = -1 (AnsiString Constant)}
  lock    dec dword ptr [ecx-8]   {Decrement Existing Reference Count}
@@NewStr:
  push   eax                     {ABS(Value)}
  mov    eax, esi                {Length}
{$ifdef UNICODE}
  mov    edx,CP_UTF8 // UTF-8 code page for Delphi 2009/2010
{$endif}
  call   system.@NewAnsiString
  mov    [edi], eax              {Set Result Address}
  mov    ecx, eax                {@Result}
  pop    eax                     {ABS(Value)}
@@LengthOk:
  mov    edi, [edi]              {@Result}
  sub    esi, ebp                {Digits Needed (Excluding Sign Character)}
  mov    byte ptr [edi], '-'     {Store '-' Character (May be Overwritten)}
  add    edi, ebp                {Destination of 1st Digit}
  pop    edx                     {Restore Abs(Value)}
  pop    eax
  cmp    esi, 17
  jl     @@LessThan17Digits      {Digits < 17}
  je     @@SetDigit17            {Digits = 17}
  cmp    esi, 18
  je     @@SetDigit18            {Digits = 18}
  mov    cl, '0' - 1
  mov    ebx, $a7640000          {1000000000000000000 mod $100000000}
  mov    ebp, $0de0b6b3          {1000000000000000000 div $100000000}
@@CalcDigit19:
  add    ecx, 1
  sub    eax, ebx
  sbb    edx, ebp
  jnc    @@CalcDigit19
  add    eax, ebx
  adc    edx, ebp
  mov    [edi], cl
  add    edi, 1
@@SetDigit18:
  mov    cl, '0' - 1
  mov    ebx, $5d8a0000          {100000000000000000 mod $100000000}
  mov    ebp, $01634578          {100000000000000000 div $100000000}
@@CalcDigit18:
  add    ecx, 1
  sub    eax, ebx
  sbb    edx, ebp
  jnc    @@CalcDigit18
  add    eax, ebx
  adc    edx, ebp
  mov    [edi], cl
  add    edi, 1
@@SetDigit17:
  mov    cl, '0' - 1
  mov    ebx, $6fc10000          {10000000000000000 mod $100000000}
  mov    ebp, $002386f2          {10000000000000000 div $100000000}
@@CalcDigit17:
  add    ecx, 1
  sub    eax, ebx
  sbb    edx, ebp
  jnc    @@CalcDigit17
  add    eax, ebx
  adc    edx, ebp
  mov    [edi], cl
  add    edi, 1                  {Update Destination}
  mov    esi, 16                 {Set 16 Digits Left}
@@LessThan17Digits:              {Process Next 8 Digits}
  mov    ecx, 100000000          {EDX:EAX = Abs(Value) = Dividend}
  div    ecx
  mov    ebp, eax                {Dividend DIV 100000000}
  mov    ebx, edx
  mov    eax, edx                {Dividend MOD 100000000}
  mov    edx, $51EB851F
  mul    edx
  shr    edx, 5                  {Dividend DIV 100}
  mov    eax, edx                {Set Next Dividend}
  lea    edx, [edx*4+edx]
  lea    edx, [edx*4+edx]
  shl    edx, 2                  {Dividend DIV 100 * 100}
  sub    ebx, edx                {Remainder (0..99)}
  movzx  ebx, word ptr [TwoDigitLookup+ebx*2]
  shl    ebx, 16
  mov    edx, $51EB851F
  mov    ecx, eax                {Dividend}
  mul    edx
  shr    edx, 5                  {Dividend DIV 100}
  mov    eax, edx
  lea    edx, [edx*4+edx]
  lea    edx, [edx*4+edx]
  shl    edx, 2                  {Dividend DIV 100 * 100}
  sub    ecx, edx                {Remainder (0..99)}
  or     bx, word ptr [TwoDigitLookup+ecx*2]
  mov    [edi+esi-4], ebx        {Store 4 Digits}
  mov    ebx, eax
  mov    edx, $51EB851F
  mul    edx
  shr    edx, 5                  {EDX = Dividend DIV 100}
  lea    eax, [edx*4+edx]
  lea    eax, [eax*4+eax]
  shl    eax, 2                  {EAX = Dividend DIV 100 * 100}
  sub    ebx, eax                {Remainder (0..99)}
  movzx  ebx, word ptr [TwoDigitLookup+ebx*2]
  movzx  ecx, word ptr [TwoDigitLookup+edx*2]
  shl    ebx, 16
  or     ebx, ecx
  mov    [edi+esi-8], ebx        {Store 4 Digits}
  mov    eax, ebp                {Remainder}
  sub    esi, 10                 {Digits Left - 2}
  jz     @@Last2Digits
@@SmallLoop:                     {Process Remaining Digits}
  mov    edx, $28F5C29           {((2^32)+100-1)/100}
  mov    ebx, eax                {Dividend}
  mul    edx
  mov    eax, edx                {Set Next Dividend}
  imul   edx, -200
  movzx  edx, word ptr [TwoDigitLookup+ebx*2+edx] {Dividend MOD 100 in ASCII}
  mov    [edi+esi], dx
  sub    esi, 2
  jg     @@SmallLoop             {Repeat Until Less than 2 Digits Remaining}
  jz     @@Last2Digits
  or     al , '0'                {Ascii Adjustment}
  mov    [edi], al               {Save Final Digit}
  jmp    @@Done
@@Last2Digits:
  movzx  eax, word ptr [TwoDigitLookup+eax*2]
  mov    [edi], ax               {Save Final 2 Digits}
@@Done:
  pop    esi
  pop    edi
  pop    ebx
end;

function Trim(const S: RawUTF8): RawUTF8;
asm  // fast implementation by John O'Harrow, modified for Delphi 2009/2010
  test eax,eax                   {S = nil?}
  xchg eax,edx
  jz   System.@LStrClr           {Yes, Return Empty String}
  mov  ecx,[edx-4]               {Length(S)}
  cmp  byte ptr [edx],' '        {S[1] <= ' '?}
  jbe  @@TrimLeft                {Yes, Trim Leading Spaces}
  cmp  byte ptr [edx+ecx-1],' '  {S[Length(S)] <= ' '?}
  jbe  @@TrimRight               {Yes, Trim Trailing Spaces}
  jmp  System.@LStrLAsg          {No, Result := S (which occurs most time)}
@@TrimLeft:                      {Strip Leading Whitespace}
  dec  ecx
  jle  System.@LStrClr           {All Whitespace}
  inc  edx
  cmp  byte ptr [edx],' '
  jbe  @@TrimLeft
@@CheckDone:
  cmp  byte ptr [edx+ecx-1],' '
{$ifdef UNICODE}
  jbe  @@TrimRight
  push CP_UTF8 // UTF-8 code page for Delphi 2009/2010
  call  System.@LStrFromPCharLen // we need a call, not a jmp here
  ret
{$else}
  ja   System.@LStrFromPCharLen
{$endif}
@@TrimRight:                     {Strip Trailing Whitespace}
  dec  ecx
  jmp  @@CheckDone
end;

{$endif FPC}  { these asm function had some low-level system.pas calls }

{$ifdef UNICODE}
// this Pos() is seldom used, it was decided to only define it under
// Delphi 2009/2010 (which expect such a RawUTF8 specific overloaded version)

function Pos(const substr, str: RawUTF8): Integer; overload;
begin
  Result := System.pos(RawByteString(substr),RawByteString(str));
end;
(*
asm // faster version by John O'Harrow: cut-down version of PosEx_JOH_6
  push    ebx
  cmp     eax,1
  sbb     ebx,ebx         {-1 if SubStr = '' else 0}
  sub     edx,1           {-1 if S = ''}
  sbb     ebx,0           {Negative if S = '' or SubStr = '' else 0}
  jl      @@InvalidInput
  push    edi
  push    esi
  push    ebp
  push    edx
  mov     edi,[eax-4]     {Length(SubStr)}
  mov     esi,[edx-3]     {Length(S)}
  cmp     edi,esi
  jg      @@NotFound      {Offset too High for a Match}
  test    edi,edi
  jz      @@NotFound      {Length(SubStr = 0)}
  lea     ebp,[eax+edi]   {Last Character Position in SubStr + 1}
  add     esi,edx         {Last Character Position in S}
  movzx   eax,[ebp-1]     {Last Character of SubStr}
  add     edx,edi         {Search Start Position in S for Last Character}
  mov     ah,al
  neg     edi             {-Length(SubStr)}
  mov     ecx,eax
  shl     eax,16
  or      ecx,eax         {All 4 Bytes = Last Character of SubStr}
@@MainLoop:
  add     edx,4
  cmp     edx,esi
  ja      @@Remainder     {1 to 4 Positions Remaining}
  mov     eax,[edx-4]     {Check Next 4 Bytes of S}
  xor     eax,ecx         {Zero Byte at each Matching Position}
  lea     ebx,[eax-$01010101]
  not     eax
  and     eax,ebx
  and     eax,$80808080   {Set Byte to $80 at each Match Position else $00}
  jz      @@MainLoop      {Loop Until any Match on Last Character Found}
  bsf     eax,eax         {Find First Match Bit}
  shr     eax,3           {Byte Offset of First Match (0..3)}
  lea     edx,[eax+edx-3] {Address of First Match on Last Character + 1}
@@Compare:
  cmp     edi,-4
  jle     @@Large         {Lenght(SubStr) >= 4}
  cmp     edi,-1
  je      @@SetResult     {Exit with Match if Lenght(SubStr) = 1}
  movzx   eax,word ptr [ebp+edi] {Last Char Matches - Compare First 2 Chars}
  cmp     ax,[edx+edi]
  jne     @@MainLoop      {No Match on First 2 Characters}
@@SetResult:              {Full Match}
  lea     eax,[edx+edi]   {Calculate and Return Result}
  pop     edx
  pop     ebp
  pop     esi
  pop     edi
  pop     ebx
  sub     eax,edx         {Subtract Start Position}
  ret
@@NotFound:
  pop     edx             {Dump Start Position}
  pop     ebp
  pop     esi
  pop     edi
@@InvalidInput:
  pop     ebx
  xor     eax,eax         {No Match Found - Return 0}
  ret
@@Remainder:              {Check Last 1 to 4 Characters}
  mov     eax,[esi-3]     {Last 4 Characters of S - May include Length Bytes}
  xor     eax,ecx         {Zero Byte at each Matching Position}
  lea     ebx,[eax-$01010101]
  not     eax
  and     eax,ebx
  and     eax,$80808080   {Set Byte to $80 at each Match Position else $00}
  jz      @@NotFound      {No Match Possible}
  lea     eax,[edx-4]     {Check Valid Match Positions}
  cmp     cl,[eax]
  lea     edx,[eax+1]
  je      @@Compare
  cmp     edx,esi
  ja      @@NotFound
  lea     edx,[eax+2]
  cmp     cl,[eax+1]
  je      @@Compare
  cmp     edx,esi
  ja      @@NotFound
  lea     edx,[eax+3]
  cmp     cl,[eax+2]
  je      @@Compare
  cmp     edx,esi
  ja      @@NotFound
  lea     edx,[eax+4]
  jmp     @@Compare
@@Large:
  mov     eax,[ebp-4]     {Compare Last 4 Characters of S and SubStr}
  cmp     eax,[edx-4]
  jne     @@MainLoop      {No Match on Last 4 Characters}
  mov     ebx,edi
@@CompareLoop:            {Compare Remaining Characters}
  add     ebx,4           {Compare 4 Characters per Loop}
  jge     @@SetResult     {All Characters Matched}
  mov     eax,[ebp+ebx-4]
  cmp     eax,[edx+ebx-4]
  je      @@CompareLoop   {Match on Next 4 Characters}
  jmp     @@MainLoop      {No Match}
end;
*)
{$endif}

{$endif LVCL}
{$endif PUREPASCAL}
{$endif ENHANCEDRTL}

function PosChar(Str: PUTF8Char; Chr: AnsiChar): PUTF8Char;
{$ifdef PUREPASCAL}
begin
  Result := Str;
  while Result^<>Chr do begin
    if Result^=#0 then begin
      Result := nil;
      Exit;
    end;
    Inc(Result);
  end;
end;
{$else}
asm // faster version by AB - eax=Str dl=Chr
    or eax,eax
    jz @z
@1: mov cl,[eax]
    cmp cl,dl
    jz @z
    inc eax
    or cl,cl
    jnz @1
    xor eax,eax
@z:
end;
{$endif}

function PosI(substr: PUTF8Char; const str: RawUTF8): Integer;
begin
  if substr<>nil then
    for result := 1 to Length(str) do
      if NormToUpper[str[result]]=substr^ then
        if IdemPChar(pointer(PtrInt(str)+result),substr+1) then
          exit;
  result := 0;
end;

procedure AppendBufferToRawUTF8(var Text: RawUTF8; Buffer: pointer; BufferLen: PtrInt);
var L: PtrInt;
begin
  if BufferLen<=0 then
    exit;
  L := PtrInt(Text);
  if L<>0 then
    L := pInteger(L-4)^; // L := length(Text)
  SetLength(Text,L+BufferLen);
  move(Buffer^,pointer(PtrInt(Text)+L)^,BufferLen);
end;


{$ifndef OWNI2S}

function Int32ToUTF8(Value : integer): RawUTF8; // faster than SysUtils.IntToStr
var tmp: array[0..15] of AnsiChar;
    P: PAnsiChar;
begin
  P := StrInt32(@tmp[15],Value);
  SetString(result,P,@tmp[15]-P);
end;

function Int64ToUtf8(Value: Int64): RawUTF8; // faster than SysUtils.IntToStr
var tmp: array[0..23] of AnsiChar;
    P: PAnsiChar;
begin
{$ifdef CPU64}
  P := StrInt32(@tmp[23],Value); // StrInt32 use PtrInt, i.e. Int64
{$else}
  P := StrInt64(@tmp[23],Value);
{$endif}
  SetString(result,P,@tmp[23]-P);
end;

{$endif}

function ExtendedToString(var S: ShortString; Value: Extended; decimals: integer): integer;
{$ifdef LVCL}
var i: integer;
{$endif}
begin
{$ifdef LVCL}
  str(Value:0:decimals,S); // not str(Value:0,S) -> '  0.0E+0000'
  // using str() here avoid FloatToStrF() usage -> LVCL is enough
  result := length(S);
  for i := 1 to result do // test if scientific format -> return as this
    if S[i]='E' then
      exit; // pos('E',S)>0; which Delphi 2009/2010 don't like
  while S[result]='0' do begin
    dec(result); // trunc any trimming 0
    if S[result]='.' then begin
      dec(result);
      break; // decimal were all '0' -> return only integer part
    end;
  end;
{$else}
{$ifdef UNICODE}
  result := RawUnicodeToUtf8(@S[1],100,
    @S[1],FloatToText(PWideChar(@S[1]), Value, fvExtended, ffGeneral, decimals, 0));
{$else}
  result := FloatToText(@S[1], Value, fvExtended, ffGeneral, decimals, 0);
{$endif}
{$endif}
end;

function ExtendedToStr(Value: Extended; decimals: integer): RawUTF8;
var tmp: ShortString;
begin
  SetString(result,PAnsiChar(@tmp[1]),ExtendedToString(tmp,Value,decimals));
end;

procedure VarRecToUTF8(const V: TVarRec; var result: RawUTF8);
begin
  with V do
  case VType of
    vtString:     result := VString^;
    vtAnsiString: result := RawUTF8(VAnsiString);
{$ifdef UNICODE}
    vtUnicodeString: result := UnicodeStringToUtf8(string(VUnicodeString));
{$endif}
    vtPChar:      result := VPChar;
    vtChar:       result := RawUTF8(VChar);
    vtWideChar:   result := RawUTF8(VWideChar);
    vtInteger:    result :=
      {$ifndef ENHANCEDRTL}Int32ToUtf8{$else}IntToStr{$endif}(VInteger);
    vtInt64:      result :=
      {$ifndef ENHANCEDRTL}Int64ToUtf8{$else}IntToStr{$endif}(VInt64^);
    vtExtended:   result := ExtendedToStr(VExtended^,10);
    else result := '';
  end;
end;

function FormatUTF8(Format: PUTF8Char; const Args: array of const): RawUTF8;
// only supported token is %, with string or integer arguments
var i, tmpN, L, A: PtrInt;
    tmp: array[0..20] of RawUTF8; // enough for 10 Args + inbetween text
    PDeb: PUTF8Char;
label Txt;
begin
  if (Format='') or (high(Args)<0) then begin
    result := Format;
    exit;
  end;
  result := '';
  tmpN := 0;
  L := 0;
  A := 0;
  while Format^<>#0 do begin
    if Format^<>'%' then begin
      PDeb := Format;
      while (Format^<>'%') and (Format^<>#0) do inc(Format);
Txt:  inc(L,Format-PDeb);
      SetString(tmp[tmpN],PDeb,Format-PDeb); // add inbetween text
      inc(tmpN);
      if tmpN=length(tmp) then
        break;
    end;
    if Format^=#0 then break;
    inc(Format); // jump '%'
    if A<=high(Args) then begin
      VarRecToUTF8(Args[A],tmp[tmpN]);
      inc(L,length(tmp[tmpN]));
      inc(A);
      inc(tmpN);
      if tmpN=length(tmp) then
        break;
    end else
    if Format^<>#0 then begin // no more available Args -> add all remaining text
      PDeb := Format;
      repeat inc(Format) until (Format^=#0);
      goto Txt;
    end;
  end;
  if L=0 then
    exit;
  SetLength(result,L);
{$ifdef UNICODE2} // not needed: SetLength() did already set the codepage
  PWord(PtrInt(result)-12)^ := CP_UTF8; // use only SetLength() -> force set code page
{$endif}
  Format := pointer(result);
  for i := 0 to tmpN-1 do
  if tmp[i]<>'' then begin
    move(pointer(tmp[i])^,Format^,PInteger(PtrInt(tmp[i])-4)^);
    inc(Format,PInteger(PtrInt(tmp[i])-4)^);
  end;  
end;

function StrIComp(Str1, Str2: PUTF8Char): integer;
{$ifdef PUREPASCAL}
var C1, C2: AnsiChar;
begin
  if Str1<>Str2 then
  if Str1<>nil then
  if Str2<>nil then begin
    repeat
      C1 := Str1^;
      C2 := Str2^;
      if C1 in ['a'..'z'] then dec(C1,32);
      if C2 in ['a'..'z'] then dec(C2,32);
      if (C1<>C2) or (C1=#0) then
        break;
      Inc(Str1);
      Inc(Str2);
    until false;
    Result := Ord(C1) - Ord(C2);
  end else
  result := 1 else  // Str2=''
  result := -1 else // Str1=''
  result := 0;      // Str1=Str2
end;
{$else}
asm // faster version by AB
        MOV     ECX,EAX
        XOR     EAX,EAX
        CMP     ECX,EDX
        JE      @Exit2  //same string or both nil
        OR      ECX,ECX
        MOV     AL,1
        JZ      @Exit2  //Str1=''
        OR      EDX,EDX
        JE      @min
@1:     MOV     AL,[ECX]
        INC     ECX
        TEST    AL,AL
        MOV     AH,[EDX]
        LEA     EDX,EDX+1
        JE      @Exit
        CMP     AL,AH
        JE      @1
        SUB     AL,'a'
        SUB     AH,'a'
        CMP     AL,'z'-'a'
        JA      @@2
        SUB     AL,20H
@@2:    CMP     AH,'z'-'a'
        JA      @@3
        SUB     AH,20H
@@3:    CMP     AL,AH
        JE      @1
@Exit:  XOR     EDX,EDX
        XCHG    AH,DL
        SUB     EAX,EDX
@Exit2: RET
@min:   OR      EAX,-1
end;
{$endif}

function StrLenW(S: PWideChar): PtrInt;
begin
  result := 0;
  if S<>nil then
  while true do
    if S[0]<>#0 then
    if S[1]<>#0 then
    if S[2]<>#0 then
    if S[3]<>#0 then begin
      inc(S,4);
      inc(result,4);
    end else begin
      inc(result,3);
      exit;
    end else begin
      inc(result,2);
      exit;
    end else begin
      inc(result);
      exit;
    end else
      exit;
end;

function StrLen(S: PUTF8Char): PtrInt;
{$ifdef PUREPASCAL}
begin
  result := 0;
  if S<>nil then
  while true do
    if S[0]<>#0 then
    if S[1]<>#0 then
    if S[2]<>#0 then
    if S[3]<>#0 then begin
      inc(S,4);
      inc(result,4);
    end else begin
      inc(result,3);
      exit;
    end else begin
      inc(result,2);
      exit;
    end else begin
      inc(result);
      exit;
    end else
      exit;
end;
{$else}
// faster than default SysUtils version
asm
     test eax,eax
     jz @@z
     cmp   byte ptr [eax+0],0; je @@0
     cmp   byte ptr [eax+1],0; je @@1
     cmp   byte ptr [eax+2],0; je @@2
     cmp   byte ptr [eax+3],0; je @@3
     push  eax
     and   eax,-4              { DWORD Align Reads }
@@Loop:
     add   eax,4
     mov   edx,[eax]           { 4 Chars per Loop }
     lea   ecx,[edx-$01010101]
     not   edx
     and   edx,ecx
     and   edx,$80808080       { Set Byte to $80 at each #0 Position }
     jz    @@Loop              { Loop until any #0 Found }
@@SetResult:
     pop   ecx
     bsf   edx,edx             { Find First #0 Position }
     shr   edx,3               { Byte Offset of First #0 }
     add   eax,edx             { Address of First #0 }
     sub   eax,ecx             { Returns Length }
@@z: ret
@@0: xor eax,eax; ret
@@1: mov eax,1;   ret
@@2: mov eax,2;   ret
@@3: mov eax,3
end;
{$endif}

function StrComp(Str1, Str2: PUTF8Char): integer;
{$ifdef PUREPASCAL}
begin
  if Str1<>Str2 then
  if Str1<>nil then
  if Str2<>nil then begin
    if Str1^=Str2^ then
    repeat
      if (Str1^=#0) or (Str2^=#0) then break;
      inc(Str1);
      inc(Str2);
    until Str1^<>Str2^;
    result := pByte(Str1)^-pByte(Str2)^;
  end else
  result := 1 else  // Str2=''
  result := -1 else // Str1=''
  result := 0;      // Str1=Str2
end;
{$else}
asm // faster version by AB
        MOV     ECX,EAX
        XOR     EAX,EAX
        CMP     ECX,EDX
        JE      @Exit2  //same string or both nil
        OR      ECX,ECX
        MOV     AL,1
        JZ      @Exit2  //Str1=''
        OR      EDX,EDX
        JE      @min
@1:     MOV     AL,[ECX]
        INC     ECX
        MOV     AH,[EDX]
        INC     EDX
        TEST    AL,AL
        JE      @Exit
        CMP     AL,AH
        JE      @1
@Exit:  XOR     EDX,EDX
        XCHG    AH,DL
        SUB     EAX,EDX
@Exit2: RET
@min:   OR      EAX,-1
end;
{$endif}


function IdemPropName(const P1,P2: shortstring): boolean; overload;
{$ifdef PUREPASCAL}
var i: integer;
begin
  result := false;
  if P1[0]<>P2[0] then
    exit;
  for i := 1 to ord(P1[0]) do
    if (ord(P1[i]) xor ord(P2[i])) and $DF<>0 then
      exit;
  result := true;
end;
{$else}
asm // eax=P1 edx=P2
    mov cl,[eax]
    cmp cl,[edx]
    jne @z      // length differs
    or cl,cl
@1: lea eax,[eax+1]
    lea edx,[edx+1]
    jz @ok
    mov ch,[eax]
    xor ch,[edx]
    and ch,$DF   // case insensitive compare
    jne @z
    dec cl
    jmp @1
@ok:mov al,1
    ret
@z: xor eax,eax
end;
{$endif}

function IdemPropName(const P1: shortstring; P2: PUTF8Char; P2Len: integer): boolean; overload;
{$ifdef PUREPASCAL}
var i: integer;
begin
  result := false;
  if ord(P1[0])<>P2Len then
    exit;
  dec(P2);
  for i := 1 to P2Len do
    if (ord(P1[i]) xor ord(P2[i])) and $DF<>0 then
      exit;
  result := true;
end;
{$else}
asm // eax=P1 edx=P2 ecx=P2Len
    cmp cl,[eax]
    jnz @z      // length differs
    or cl,cl
@1: lea eax,[eax+1]
    jz @ok
    mov ch,[edx]
    xor ch,[eax]
    and ch,$DF   // case insensitive compare
    jne @z
    dec cl
    lea edx,[edx+1]
    jmp @1
@ok:mov al,1
    ret
@z: xor eax,eax
end;
{$endif}

procedure InitSynCommonsConversionTables;
var i: integer;
{$ifdef OWNNORMTOUPPER}
   c, d: AnsiChar;
const n2u: array[138..255] of byte =
  (83,139,140,141,90,143,144,145,146,147,148,149,150,151,152,153,83,155,140,
   157,90,89,160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,
   176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,65,65,65,
   65,65,65,198,67,69,69,69,69,73,73,73,73,68,78,79,79,79,79,79,215,79,85,85,
   85,85,89,222,223,65,65,65,65,65,65,198,67,69,69,69,69,73,73,73,73,68,78,79,
   79,79,79,79,247,79,85,85,85,85,89,222,89);
{$endif OWNNORMTOUPPER}
begin
{$ifdef OWNNORMTOUPPER}
  // initialize custom NormToUpper[] and NormToLower[] arrays
  for c := #0 to #137 do
    NormToUpper[c] := c;
  for c := 'a' to 'z' do
    dec(NormToUpper[c],ord('a')-ord('A'));
  move(n2u,NormToUpper[#138],sizeof(n2u));
  for c := #0 to #255 do begin
    d := NormToUpper[c];
    if d in ['A'..'Z'] then
      inc(d,32);
    NormToLower[c] := d;
  end;
{$endif OWNNORMTOUPPER}
  for i := 0 to 255 do
    if (i>=low(WinAnsiUnicodeChars)) and (i<=high(WinAnsiUnicodeChars)) then
      WinAnsiTable[i] := WinAnsiUnicodeChars[i] else
      WinAnsiTable[i] := i;
end;

procedure InitConvertHexToBin;
var v: byte;
    i: integer;
begin // code below is 55 bytes long, therefore shorter than a const array
  if Pointer(ConvertHexToBin)<>nil then
    exit;
  SetLength(ConvertHexToBin,256);
  fillchar(ConvertHexToBin[0],sizeof(ConvertHexToBin),255); // all to 255
  v := 0;
  for i := ord('0') to ord('9') do begin
    ConvertHexToBin[i] := v;
    inc(v);
  end;
  for i := ord('A') to ord('F') do begin
    ConvertHexToBin[i] := v;
    ConvertHexToBin[i+(ord('a')-ord('A'))] := v;
    inc(v);
  end;
end;

procedure SoundExComputeAnsi(var p: PAnsiChar; var result: cardinal; Values: PSoundExValues);
var n,v,old: cardinal;
begin
  n := 0;
  old := 0;
  if Values<>nil then
  repeat
{$ifdef USENORMTOUPPER}
    v := NormToUpperByte[ord(p^)]; // also handle 
{$else}
    v := ord(p^);
    if v in [ord('a')..ord('z')] then
      dec(v,32); // approximative 7 bit char uppercase
{$endif}
    if not (v in IsWord) then break;
    inc(p);
    dec(v,ord('B'));
    if v>high(TSoundExValues) then continue;
    v := Values[v]; // get soundex value
    if (v=0) or (v=old) then continue; // invalid or dopple value
    old := v;
    result := result shl SOUNDEX_BITS;
    inc(result,v);
    inc(n);
    if n=((32-8)div SOUNDEX_BITS) then // first char use up to 8 bits
      break; // result up to a cardinal size
  until false;
end;

function SoundExComputeFirstCharAnsi(var p: PAnsiChar): cardinal;
label Err;
begin
  if p=nil then begin
Err:result := 0;
    exit;
  end;
  repeat
{$ifdef USENORMTOUPPER}
    result := NormToUpperByte[ord(p^)]; // also handle 
{$else}
    result := ord(p^);
    if result in [ord('a')..ord('z')] then
      dec(result,32); // approximative 7 bit char uppercase
{$endif}
    if result=0 then
      goto Err; // end of input text, without a word
    inc(p);
    // trim initial spaces or 'H'
  until AnsiChar(result) in ['A'..'G','I'..'Z'];
end;

function GetNextUTF8Upper(var U: PUTF8Char): cardinal;
begin
  result := ord(U^);
  if result<>0 then
  if result and $80=0 then begin
    inc(U);
{$ifdef USENORMTOUPPER}
    result := NormToUpperByte[result]; {$else}
    if result in [ord('a')..ord('z')] then
      dec(result,32); // approximative 7 bit char uppercase
{$endif}
    exit;
  end else
  if result and $20=0 then begin
    result := result shl 6+byte(U[1])-$00003080;
    inc(U,2);
    if result<=255 then begin
{$ifdef USENORMTOUPPER}
      result := NormToUpperByte[result]; {$else}
      if result in [ord('a')..ord('z')] then
        dec(result,32); // approximative 7 bit char uppercase
{$endif}
    end else
      result := ord('?'); // char ignored for soundex
    exit;
  end else begin
    inc(U,3);
    result := ord('?');   // char ignored for soundex
    exit;
  end else
    exit;
end;

procedure SoundExComputeUTF8(var U: PUTF8Char; var result: cardinal; Values: PSoundExValues);
var n,v,old: cardinal;
begin
  n := 0;
  old := 0;
  if Values<>nil then
  repeat
    v := GetNextUTF8Upper(U);
    if not (v in IsWord) then break;
    dec(v,ord('B'));
    if v>high(TSoundExValues) then continue;
    v := Values[v]; // get soundex value
    if (v=0) or (v=old) then continue; // invalid or dopple value
    old := v;
    result := result shl SOUNDEX_BITS;
    inc(result,v);
    inc(n);
    if n=((32-8)div SOUNDEX_BITS) then // first char use up to 8 bits
      break; // result up to a cardinal size
  until false;
end;

function SoundExComputeFirstCharUTF8(var U: PUTF8Char): cardinal;
label Err;
begin
  if U=nil then begin
Err:result := 0;
    exit;
  end;
  repeat
    result := GetNextUTF8Upper(U);
    if result=0 then
      goto Err; // end of input text, without a word
    // trim initial spaces or 'H'
  until AnsiChar(result) in ['A'..'G','I'..'Z'];
end;

function FindNextUTF8WordBegin(U: PUTF8Char): PUTF8Char;
{$ifdef USENORMTOUPPER}
var c: cardinal;
begin
  result := nil;
  repeat
    c := byte(U^);
    if c=0 then
      exit else
    if c and $80=0 then
      if not (c in IsWord) then
        break else
        inc(U) else
    if c and $20=0 then begin
      c := c shl 6+byte(U[1])-$00003080;
      if (c<=255) and not (NormToUpperByte[c] in IsWord) then
        break;
      inc(U,2);
    end else
      inc(U,3);
  until false;
  result := U;
end;
{$else}
begin
  repeat
    if byte(U^)=0 then begin
      result := nil;
      exit;
    end else
    if byte(U^) and $80=0 then
      if not (byte(U^) in isWord) then
        break else
        inc(U) else
    if byte(U^) and $20=0 then
      inc(U,2) else
      inc(U,3);
  until false;
  result := U;
end;
{$endif}

{ TSynSoundEx }

const
  /// english Soundex pronunciation scores
  // - defines the default values used for the SoundEx() function below
  // (used if Values parameter is nil)
  ValueEnglish: TSoundExValues =
  // B C D E F G H I J K L M N O P Q R S T U V W X Y Z
    (1,2,3,0,1,2,0,0,2,2,4,5,5,0,1,2,6,2,3,0,1,0,2,0,2);

  /// french Soundex pronunciation scores
  // - can be used to override default values used for the SoundEx()
  // function below
  ValueFrench: TSoundExValues =
  // B C D E F G H I J K L M N O P Q R S T U V W X Y Z
    (1,2,3,0,9,7,0,0,7,2,4,5,5,0,1,2,6,8,3,0,9,0,8,0,8);

  /// spanish Soundex pronunciation scores
  // - can be used to override default values used for the SoundEx()
  // function below
  ValueSpanish: TSoundExValues =
  // B C D E F G H I J K L M N O P Q R S T U V W X Y Z
    (1,2,3,0,1,2,0,0,0,2,0,5,5,0,1,2,6,2,3,0,1,0,2,0,2);

  SOUNDEXVALUES: array[TSynSoundExPronunciation] of PSoundExValues =
    (@ValueEnglish,@ValueFrench,@ValueSpanish,@ValueEnglish);

function TSynSoundEx.Ansi(A: PAnsiChar): boolean;
var Value, c: cardinal;
begin
  result := false;
  if A=nil then exit;
  repeat
    // test beginning of word
    c := SoundExComputeFirstCharAnsi(A);
    if c=0 then exit else
    if c=FirstChar then begin
      // here we had the first char match -> check if word match UpperValue
      Value := c-(ord('A')-1);
      SoundExComputeAnsi(A,Value,fValues);
      if Value=search then begin
        result := true; // UpperValue found!
        exit;
      end;
    end;
    // find beginning of next word
    repeat
      if A^=#0 then exit else
{$ifdef USENORMTOUPPER}
        if not (NormToUpperByte[ord(A^)] in IsWord) then break else inc(A);
{$else} if not (ord(A^) in IsWord) then break else inc(A); {$endif}
    until false;
  until false;
end;

function TSynSoundEx.UTF8(U: PUTF8Char): boolean;
var Value, c: cardinal;
begin
  result := false;
  if U=nil then exit;
  repeat
    // find beginning of word
    c := SoundExComputeFirstCharUTF8(U);
    if c=0 then exit else
    if c=FirstChar then begin
      // here we had the first char match -> check if word match UpperValue
      Value := c-(ord('A')-1);
      SoundExComputeUTF8(U,Value,fValues);
      if Value=search then begin
        result := true; // UpperValue found!
        exit;
      end;
    end;
    // find beginning of next word
    U := FindNextUTF8WordBegin(U);
  until U=nil;
end;

function TSynSoundEx.Prepare(UpperValue: PAnsiChar; Lang: TSynSoundExPronunciation): boolean;
begin
  fValues := SOUNDEXVALUES[Lang];
  Search := SoundExAnsi(UpperValue,nil,Lang);
  if Search=0 then
    result := false else begin
    FirstChar := SoundExComputeFirstCharAnsi(UpperValue);
    result := true;
  end;
end;

function SoundExAnsi(A: PAnsiChar; next: PPAnsiChar;
  Lang: TSynSoundExPronunciation): cardinal;
begin
  result := SoundExComputeFirstCharAnsi(A);
  if result<>0 then begin
    dec(result,ord('A')-1);   // first Soundex char is first char
    SoundExComputeAnsi(A,result,SOUNDEXVALUES[Lang]);
  end;
  if next<>nil then begin
{$ifdef USENORMTOUPPER}
    while NormToUpperByte[ord(A^)] in IsWord do inc(A); // go to end of word
{$else}
    while ord(A^) in IsWord do inc(A); // go to end of word
{$endif}
    next^ := A;
  end;
end;

function SoundExUTF8(U: PUTF8Char; next: PPUTF8Char;
  Lang: TSynSoundExPronunciation): cardinal;
begin
  result := SoundExComputeFirstCharUTF8(U);
  if result<>0 then begin
    dec(result,ord('A')-1);   // first Soundex char is first char
    SoundExComputeUTF8(U,result,SOUNDEXVALUES[Lang]);
  end;
  if next<>nil then
    next^ := FindNextUTF8WordBegin(U);
end;

{$ifdef USENORMTOUPPER}

function AnsiICompW(u1, u2: PWideChar): PtrInt;
begin
  if u1<>u2 then
    if u1<>nil then
      if u2<>nil then
        repeat
          result := PtrInt(u1^)-PtrInt(u2^);
          if result<>0 then begin
            if (PtrInt(u1^)>255) or (PtrInt(u2^)>255) then exit;
            result := NormToUpperByte[PtrInt(u1^)]-NormToUpperByte[PtrInt(u2^)];
            if result<>0 then exit;
          end;
          if (u1^=#0) or (u2^=#0) then break;
          inc(u1);
          inc(u2);
        until false else
        result := 1 else  // u2=''
      result := -1 else // u1=''
    result := 0;      // u1=u2
end;


function AnsiIComp(Str1, Str2: PUTF8Char): PtrInt;
{$ifdef PUREPASCAL}
begin
  if Str1<>Str2 then
  if Str1<>nil then
  if Str2<>nil then
  repeat
    result := NormToUpperByte[ord(Str1^)]-NormToUpperByte[pByte(Str2)^];
    if result<>0 then exit;
    if (Str1^=#0) or (Str2^=#0) then break;
    inc(Str1);
    inc(Str2);
  until false else
  result := 1 else  // Str2=''
  result := -1 else // Str1=''
  result := 0;      // Str1=Str2
end;
{$else}
asm // fast 8 bits WinAnsi comparaison using the NormToUpper[] array
    cmp eax,edx
    je @2
    test eax,edx // is either of the strings perhaps nil?
    jz @3
@0: push ebx // compare the first character (faster quicksort)
    movzx ebx,byte ptr [eax] // ebx=S1[1]
    movzx ecx,byte ptr [edx] // ecx=S2[1]
    or ebx,ebx
    jz @z
    cmp ebx,ecx
    je @s
    mov bl,byte ptr [NormToUpper+ebx]
    mov cl,byte ptr [NormToUpper+ecx]
    cmp ebx,ecx
    je @s
    mov eax,ebx
    pop ebx
    sub eax,ecx // return S1[1]-S2[1]
    ret
@2: xor eax, eax
    ret
@3: test eax,eax // S1=''
    jz @4
    test edx,edx // S2='' ?
    jnz @0
    mov eax,1 // return 1 (S1>S2)
    ret
@s: inc eax
    inc edx
    mov bl,[eax] // ebx=S1[i]
    mov cl,[edx] // ecx=S2[i]
    or ebx,ebx
    je @z        // end of S1
    cmp ebx,ecx
    je @s
    mov bl,byte ptr [NormToUpper+ebx]
    mov cl,byte ptr [NormToUpper+ecx]
    cmp ebx,ecx
    je @s
    mov eax,ebx
    pop ebx
    sub eax,ecx // return S1[i]-S2[i]
    ret
@z: cmp ebx,ecx // S1=S2?
    pop ebx
    jz @2
@4: or eax,-1 // return -1 (S1<S2)
end;
{$endif}

function UTF8IComp(u1, u2: PUTF8Char): PtrInt;
var c2: PtrInt;
begin // fast UTF-8 comparaison using the NormToUpper[] array for all 8 bits values
  if u1<>u2 then
  if u1<>nil then
  if u2<>nil then
  repeat
    result := pbyte(u1)^;
    inc(u1);
    if result and $80=0 then
      if result<>0 then
        result := NormToUpperByte[result] else begin // end of u1 reached
        if u2^<>#0 then    // end of u2 reached -> u1=u2 -> return 0
          result := -1;    // u1<u2
        exit;
      end else begin
        if result and $20=0 then begin
          result := result shl 6+pbyte(u1)^-$00003080;
          inc(u1);
        end else begin
          result := result shl 6+pbyte(u1)^;
          inc(u1);
          result := result shl 6+pbyte(u1)^-$000E2080;
          inc(u1);
        end;
        if result and $ffffff00=0 then
          result := NormToUpperByte[result]; // 8 bits to upper, 32 bits as is
      end;
    c2 := pbyte(u2)^; inc(u2);
    if c2 and $80=0 then begin
      if c2=0 then exit; // u1>u2 -> return u1^
      dec(result,NormToUpperByte[c2]);
      if result<>0 then exit;
    end else begin
      if c2 and $20=0 then begin
        c2 := c2 shl 6+pbyte(u2)^-$00003080;
        inc(u2);
      end else begin
        c2 := c2 shl 6+pbyte(u2)^;
        inc(u2);
        c2 := c2 shl 6+pbyte(u2)^-$000E2080;
        inc(u2);
      end;
      if c2 and $ffffff00=0 then
        dec(result,NormToUpperByte[c2]) else // 8 bits to upper
        dec(result,c2); // 32 bits widechar returns diff
      if result<>0 then exit;
    end;
  until false else
  result := 1 else  // u2=''
  result := -1 else // u1=''
  result := 0;      // u1=u2
end;

function UTF8ILComp(u1, u2: PUTF8Char; L1,L2: cardinal): PtrInt;
var c2: PtrInt;
label neg,pos;
begin // fast UTF-8 comparaison using the NormToUpper[] array for all 8 bits values
  if u1<>u2 then
  if (u1<>nil) and (L1<>0) then
  if (u2<>nil) and (L2<>0) then
  repeat
    result := pbyte(u1)^;
    inc(u1);
    dec(L1);
    if result and $80=0 then
      result := NormToUpperByte[result] else begin
      if L1=0 then goto neg else dec(L1);
      if result and $20=0 then begin
        result := result shl 6+pbyte(u1)^-$00003080;
        inc(u1);
      end else begin
        if L1=0 then goto neg else dec(L1);
        result := result shl 6+pbyte(u1)^;
        inc(u1);
        result := result shl 6+pbyte(u1)^-$000E2080;
        inc(u1);
      end;
      if result and $ffffff00=0 then
        result := NormToUpperByte[result]; // 8 bits to upper, 32 bits as is
    end;
    // here result=NormToUpper[u1^]
    c2 := pbyte(u2)^;
    inc(u2);
    dec(L2);
    if c2 and $80=0 then begin
      dec(result,NormToUpperByte[c2]);
      if result<>0 then exit;
    end else begin
      if L2=0 then goto pos else dec(L2);
      if c2 and $20=0 then begin
        c2 := c2 shl 6+pbyte(u2)^-$00003080;
        inc(u2);
      end else begin
        if L2=0 then goto pos else dec(L2);
        c2 := c2 shl 6+pbyte(u2)^;
        inc(u2);
        c2 := c2 shl 6+pbyte(u2)^-$000E2080;
        inc(u2);
      end;
      if c2 and $ffffff00=0 then
        dec(result,NormToUpperByte[c2]) else // 8 bits to upper
        dec(result,c2); // returns 32 bits diff
      if result<>0 then exit;
    end;
    // here we have result=NormToUpper[u2^]-NormToUpper[u1^]=0
    if L1=0 then // test if we reached end of u1 or end of u2
      if L2=0 then exit     // u1=u2
         else goto neg else // u1<u2
    if L2=0 then goto pos;  // u1>u2
  until false else
pos: result := 1 else  // u2='' or u1>u2
neg: result := -1 else // u1='' or u1<u2
     result := 0;      // u1=u2
end;

{$define SAMETEXTCORRECT}
// we expect real UTF-8 decoding

function SameTextU(const S1, S2: RawUTF8): Boolean;
{$ifdef SAMETEXTCORRECT}
// code used NormToUpper[] after UTF-8 decoding
{$ifdef PUREPASCAL}
var L: PtrInt;
begin
  L := length(S2);
  if length(S1)<>L then
    result := false else
    result := UTF8IComp(pointer(S1),pointer(S2))=0;
end;
{$else}
asm
      cmp eax,edx
      jz @ok
      or eax,eax
      jz @ret
      or edx,edx
      jz @no
      mov ecx,[eax-4]
      cmp ecx,[edx-4] // length(s1)=length(s2) ?
      jne @no
      // eax=s1 edx=s2
      call UTF8IComp
      test eax,eax
@ok:  setz al
@ret: ret
@no:  xor eax,eax
end;
{$endif}
{$else}
// this code use NormToUpper[] directly, with no UTF-8 decoding: buggy
{$ifdef PUREPASCAL}
var L, i: PtrInt;
begin
  L := length(S2);
  if length(S1)<>L then begin
    result := false;
    exit;
  end;
  for i := 0 to L-1 do
    if NormToUpper[S1[i+1]]<>NormToUpper[S2[i+1]] then begin
      result := false;
      exit;
    end;
  result := true;
end;
{$else}
asm
        cmp eax,edx
        jz @ok
        or eax,eax
        jz @ret
        or edx,edx
        jz @no
        mov ecx,[eax-4]
        cmp ecx,[edx-4] // length(s1)=length(s2) ?
        jne @no
        // eax=s1 edx=s2
        push ebx
        xor ecx,ecx
        xor ebx,ebx
@loop:  mov cl,[eax] // cl = s1^
        mov bl,[edx] // bl = s2^
        lea eax,[eax+1]
        test ecx,ecx
        mov cl,[ecx+NormToUpper]
        jz @okloop // s1^=#0 -> same text -> return true
        cmp cl,[ebx+NormToUpper]
        lea edx,[edx+1]
        je @loop
        pop ebx
@no:    xor eax,eax
@ret:   ret
@okloop:pop ebx
@ok:    mov al,1
end;
{$endif}
{$endif}

{$else} // no NormToUpper[]

function AnsiIComp(Str1, Str2: PUTF8Char): integer;
{$ifdef PUREPASCAL}
begin
  result := StrIComp(Str1,Str2); // fast enough
end;
{$else}
asm
  jmp StrIComp // LVCL without NormToUpper[]: use default SysUtils implementation
end;
{$endif}
{$endif}

function FindAnsi(A, UpperValue: PAnsiChar): boolean;
var ValueStart: PAnsiChar;
{$ifndef USENORMTOUPPER}
    ch: AnsiChar;
{$endif}
begin
  result := false;
  if (A=nil) or (UpperValue=nil) then exit;
  ValueStart := UpperValue;
  repeat
    // test beginning of word
    repeat
      if A^=#0 then exit else
{$ifdef USENORMTOUPPER}
      if byte(NormToUpper[A^]) in IsWord then break else inc(A);
{$else}
      ch := A^;
      if ch in ['a'..'z'] then
        dec(ch,32); // 7 bits uppercase
      if byte(ch) in IsWord then break else inc(A);
{$endif}
    until false;
    // check if this word is the UpperValue
    UpperValue := ValueStart;
    repeat
{$ifdef USENORMTOUPPER}
      if NormToUpper[A^]<>UpperValue^ then break;
{$else}
      ch := A^;
      if ch in ['a'..'z'] then
        dec(ch,32); // 7 bits uppercase
      if ch<>UpperValue^ then break;
{$endif}
      inc(UpperValue);
      if UpperValue^=#0 then begin
        result := true; // UpperValue found!
        exit;
      end;
      inc(A);
      if A^=#0 then exit;
    until false;
    // find beginning of next word
    repeat
      if A^=#0 then exit else
{$ifdef USENORMTOUPPER}
        if not (NormToUpperByte[ord(A^)] in IsWord) then break else inc(A);
{$else} if not (ord(A^) in IsWord) then break else inc(A); {$endif}
    until false;
  until false;
end;

function FindUTF8(U: PUTF8Char; UpperValue: PAnsiChar): boolean;
var ValueStart: PAnsiChar;
{$ifdef USENORMTOUPPER}
    c: cardinal;
    FirstChar: AnsiChar;
label Next;
{$else}
    ch: AnsiChar;
{$endif}
begin
  result := false;
  if (U=nil) or (UpperValue=nil) then exit;
{$ifdef USENORMTOUPPER}
  // handles 8-bits WinAnsi chars inside UTF-8 encoded data
  FirstChar := UpperValue^;
  ValueStart := UpperValue+1;
  repeat
    // test beginning of word
    repeat
      c := byte(U^); inc(U);
      if c=0 then exit else
      if c and $80=0 then begin
        if c in IsWord then
          if PAnsiChar(@NormToUpper)[c]<>FirstChar then
            goto Next else
            break;
      end else
      if c and $20=0 then begin
        c := c shl 6+byte(U^)-$00003080;
        inc(U);
        if c<=255 then begin
          c := NormToUpperByte[c];
          if c in IsWord then
            if AnsiChar(c)<>FirstChar then
              goto Next else
              break;
        end;
      end else
        inc(U,2);
    until false;
    // here we had the first char match -> check if this word match UpperValue
    UpperValue := ValueStart;
    repeat
      if UpperValue^=#0 then begin
        result := true; // UpperValue found!
        exit;
      end;
      c := byte(U^); inc(U); // next chars
      if c=0 then exit else
      if c and $80=0 then begin
        if PAnsiChar(@NormToUpper)[c]<>UpperValue^ then break;
      end else
      if c and $20=0 then begin
        c := c shl 6+byte(U^)-$00003080;
        inc(U);
        if (c>255) or (PAnsiChar(@NormToUpper)[c]<>UpperValue^) then break;
      end else begin
        inc(U,2);
        break;
      end;
      inc(UpperValue);
    until false;
    // find beginning of next word
Next:
{$else}
  // this tiny version only handles 7-bits ansi chars and ignore all UTF-8 chars
  ValueStart := UpperValue;
  repeat
    // find beginning of word
    repeat
      if byte(U^)=0 then exit else
      if byte(U^) and $80=0 then
        if byte(U^) in IsWord then
          break else
          inc(U) else
      if byte(U^) and $20=0 then
        inc(U,2) else
        inc(U,3);
    until false;
    // check if this word is the UpperValue
    UpperValue := ValueStart;
    repeat
      ch := U^;
      if ch in ['a'..'z'] then
        dec(ch,32); // 7 bits uppercase
      if ch<>UpperValue^ then break;
      inc(UpperValue);
      if UpperValue^=#0 then begin
        result := true; // UpperValue found!
        exit;
      end;
      inc(U);
      if byte(U^)=0 then exit else
      if byte(U^) and $80<>0 then break; // 7 bits char check only
    until false;
{$endif}
    // find beginning of next word
    U := FindNextUTF8WordBegin(U);
  until U=nil;
end;

function HexToBin(Hex: PAnsiChar; Bin: PByte; BinBytes: Integer): boolean;
var I: Integer;
    B,C: byte;
begin
  result := false; // return false if any invalid char
  if Pointer(ConvertHexToBin)=nil then
    InitConvertHexToBin;
  if Bin<>nil then
  for I := 1 to BinBytes do begin
    B := ConvertHexToBin[Ord(Hex^)];
    inc(Hex);
    if B>15 then exit;
    C := ConvertHexToBin[Ord(Hex^)];
    Inc(Hex);
    if C>15 then exit;
    Bin^ := B shl 4+C;
    Inc(Bin);
  end else
  for I := 1 to BinBytes do begin // no Bin^ -> just validate Hex^ Stream format
    B := ConvertHexToBin[Ord(Hex^)];
    inc(Hex);
    if B>15 then exit;
    C := ConvertHexToBin[Ord(Hex^)];
    Inc(Hex);
    if C>15 then exit;
  end;
  result := true; // conversion OK
end;

const
  /// fast lookup table for converting hexadecimal numbers from 0 to 15
  // into their ASCII equivalence
  // - our enhanced SysUtils.pas (normal and LVCL) contains the same array
  // - should be local for better code generation
  HexChars: array[0..15] of AnsiChar = '0123456789ABCDEF';


procedure BinToHex(Bin, Hex: PAnsiChar; BinBytes: integer);
var j, v: cardinal;
begin
  for j := 1 to BinBytes do begin
    v := byte(Bin^);
    Hex[0] := HexChars[v shr 4];
    Hex[1] := HexChars[v and $F];
    inc(Hex,2);
    inc(Bin);
  end;
end;

function BinToHex(const Bin: RawByteString): RawUTF8; overload;
var L: integer;
begin
  L := length(Bin);
  SetString(result,nil,L*2);
  SynCommons.BinToHex(pointer(Bin),pointer(Result),L);
end;

procedure YearToPChar(Y: Word; P: PUTF8Char);
{$ifdef PUREPASCAL}
begin
  PWord(P  )^ := TwoDigitLookupW[Y div 100];
  PWord(P+2)^ := TwoDigitLookupW[Y mod 100];
end;
{$else}
asm
  mov cl,100
  div cl // ah=remainder=Y mod 100, al=quotient=Year div 100
  movzx ecx,al // al=quotient=Y div 100
  mov cx,word ptr [TwoDigitLookup+ecx*2]
  mov [edx],cx
  movzx ecx,ah // ah=remainder=Y mod 100
  mov cx,word ptr [TwoDigitLookup+ecx*2]
  mov [edx+2],cx
end;
{$endif}

function SameValue(const A, B: Double; DoublePrec: double = 1E-10): Boolean;
var AbsA,AbsB: double;
begin // faster than the Math unit version
  AbsA := Abs(A);
  AbsB := Abs(B);
  if AbsA<AbsB then
    AbsA := AbsA*DoublePrec else
    AbsA := AbsB*DoublePrec; // AbsA := Min(Abs(A),Abs(B))*DoublePrec
  // AbsA is the allowed Epsilon value
  if AbsA<DoublePrec then
    Result := Abs(A-B)<=DoublePrec else
    Result := Abs(A-B)<=AbsA;
end;

/// return the index of Value in Values[], -1 if not found
function FindRawUTF8(const Values: TRawUTF8DynArray; const Value: RawUTF8;
  CaseSensitive: boolean=true): integer;
begin
  if CaseSensitive then begin
    for result := 0 to high(Values) do
      if Values[result]=Value then
        exit;
  end else
    for result := 0 to high(Values) do
      if SameTextU(Values[result],Value) then
        exit;
  result := -1;
end;

/// true if Value was added successfully in Values[]
function AddRawUTF8(var Values: TRawUTF8DynArray; const Value: RawUTF8;
  NoDupplicates: boolean=false; CaseSensitive: boolean=true): boolean;
var i: integer;
begin
  if NoDupplicates then begin
    i := FindRawUTF8(Values,Value,CaseSensitive);
    if i>=0 then begin
      result := false;
      exit;
    end;
  end;
  i := length(Values);
  SetLength(Values,i+1);
  Values[i] := Value;
  result := true;
end;

/// find the position of the SEARCH] section in source
// - return true if SEARCH] was found, and store line after it in source
function FindSectionFirstLine(var source: PUTF8Char; search: PUTF8Char): boolean;
{$ifdef PUREPASCAL}
begin
  result := false;
  if source=nil then
    exit;
  repeat
    if source^='[' then begin
      inc(source);
      result := IdemPChar(source,search);
    end;
    while not (source^ in [#0,#10,#13]) do inc(source);
    while source^ in [#10,#13] do inc(source);
    if result then
      exit; // found
  until source^=#0;
  source := nil;
end;
{$else}
asm // eax=source edx=search
    push eax       // save source var
    mov eax,[eax]  // eax=source
    or eax,eax
    jz @z
    push ebx
    mov ebx,edx    // save search
    cmp byte ptr [eax],'['
    lea eax,eax+1
    jne @s
@i: push eax
    mov edx,ebx   // edx=search
    call IdemPChar
    pop ecx       // ecx=source
    jmp @1
@s: mov ecx,eax
    xor eax,eax   // result := false
@1: mov dl,[ecx]  // while not (source^ in [#0,#10,#13]) do inc(source);
    inc ecx
    cmp dl,13
    ja @1
    je @e
    or dl,dl
    jz @0
    cmp dl,10
    jne @1
    jmp @4
@e: cmp byte ptr [ecx],10 // jump #13#10
    jne @4
    inc ecx
@4: test al,al
    jnz @x         // exit if IdemPChar returned true
    cmp byte ptr [ecx],'['
    lea ecx,ecx+1
    jne @1
    mov eax,ecx
    jmp @i
@0: xor ecx,ecx    // set source=nil
@x: pop ebx
    pop edx        // restore source var
    mov [edx],ecx  // update source var
    ret
@z: pop edx       // ignore source var, result := false
end;
{$endif}

function IdemPCharW(p: pWideChar; up: PUTF8Char): boolean;
// if the beginning of p^ is same as up^ (ignore case - up^ must be already Upper)
{$ifdef USENORMTOUPPER}
{$ifdef PUREPASCAL}
begin
  result := false;
  if (p=nil) or (up=nil) then
    exit;
  while up^<>#0 do begin
    if (p^>#255) or (up^<>AnsiChar(NormToUpperByte[ord(p^)])) then
      exit;
    inc(up);
    inc(p);
  end;
  result := true;
end;
{$else}
// eax=p edx=up
asm
  or eax,eax
  jz @e // P=nil -> false
  or edx,edx
  push ebx
  push esi
  jz @z // up=nil -> true
  mov esi,offset NormToUpper
  xor ebx,ebx
  xor ecx,ecx
@1:
  mov bx,[eax] // bl=p^
  mov cl,[edx] // cl=up^
  or bh,bh     // p^ > #255 -> FALSE
  jnz @n
  test cl,cl
  mov bl,[ebx+esi] // bl=NormToUpper[p^]
  jz @z // up^=#0 -> OK
  lea edx,[edx+1] // = inc edx without changing flags
  cmp bl,cl
  lea eax,[eax+2]
  je @1
@n:
  pop esi
  pop ebx
  xor eax,eax
@e:
  ret
@z:
  mov al,1 // up^=#0 -> OK
  pop esi
  pop ebx
end;
{$endif}
{$else}
var c: WideChar;
begin
  result := false;
  if (p=nil) or (up=nil) then
    exit;
  while up^<>#0 do begin
    c := p^;
    if ord(up^)<>ord(c) then
      if c in [WideChar('a')..WideChar('z')] then begin
        dec(c,32);
        if ord(up^)<>ord(c) then
          exit;
      end else exit;
    inc(up);
    inc(p);
  end;
  result := true;
end;
{$endif}

function FindSectionFirstLineW(var source: PWideChar; search: PUTF8Char): boolean;
{$ifdef PUREPASCAL}
begin
  result := false;
  if source=nil then
    exit;
  repeat
    if source^='[' then begin
      inc(source);
      result := IdemPCharW(source,search);
    end;
    while not (cardinal(source^) in [0,10,13]) do inc(source);
    while cardinal(source^) in [10,13] do inc(source);
    if result then
      exit; // found
  until source^=#0;
  source := nil;
end;
{$else}
asm // eax=source edx=search
    push eax       // save source var
    mov eax,[eax]  // eax=source
    or eax,eax
    jz @z
    push ebx
    mov ebx,edx    // save search
    cmp word ptr [eax],'['
    lea eax,eax+2
    jne @s
@i: push eax
    mov edx,ebx   // edx=search
    call IdemPCharW
    pop ecx       // ecx=source
    jmp @1
@s: mov ecx,eax
    xor eax,eax   // result := false
@1: mov dx,[ecx]  // while not (source^ in [#0,#10,#13]) do inc(source);
    lea ecx,ecx+2
    cmp dx,13
    ja @1
    je @e
    or dx,dx
    jz @0
    cmp dx,10
    jne @1
    jmp @4
@e: cmp word ptr [ecx],10 // jump #13#10
    jne @4
    lea ecx,ecx+2
@4: test al,al
    jnz @x         // exit if IdemPChar returned true
    cmp word ptr [ecx],'['
    lea ecx,ecx+2
    jne @1
    mov eax,ecx
    jmp @i
@0: xor ecx,ecx    // set source=nil
@x: pop ebx
    pop edx        // restore source var
    mov [edx],ecx  // update source var
    ret
@z: pop edx       // ignore source var, result := false
end;
{$endif}

/// find the Value of UpperName in P, till end of current section
// - expect UpperName as 'NAME='
function FindIniNameValue(P: PUTF8Char; UpperName: PUTF8Char): RawUTF8;
var PDeb: PUTF8Char;
    L: integer;
begin
  while (P<>nil) and (P^<>'[') do begin
    PDeb := GetNextLineBegin(P,P); // since PDeb=P, we have PDeb<>nil
    if PDeb^=' ' then repeat inc(PDeb) until PDeb^<>' ';   // trim left ' '
    if IdemPChar(PDeb,UpperName) then begin
      inc(PDeb,StrLen(UpperName));
      L := 0; while PDeb[L]>=' ' do inc(L); // get line length
      SetString(result,PDeb,L);
      exit;
    end;
  end;
  result := '';
end;

function ExistsIniNameValue(P: PUTF8Char; UpperName: PUTF8Char): boolean;
var PDeb: PUTF8Char;
begin
  result := true;
  while (P<>nil) and (P^<>'[') do begin
    PDeb := GetNextLineBegin(P,P); // since PDeb=P, we have PDeb<>nil
    if PDeb^=' ' then repeat inc(PDeb) until PDeb^<>' ';   // trim left ' '
    if IdemPChar(PDeb,UpperName) then
      exit;
  end;
  result := false;
end;

function FindWinAnsiIniNameValue(P: PUTF8Char; UpperName: PUTF8Char): RawUTF8;
begin
  result := WinAnsiToUtf8(WinAnsiString(FindIniNameValue(P,UpperName)));
end;

function ExistsIniName(P: PUTF8Char; UpperName: PUTF8Char): boolean;
var PDeb: PUTF8Char;
begin
  result := true;
  while (P<>nil) and (P^<>'[') do begin
    PDeb := GetNextLineBegin(P,P); // since PDeb=P, we have PDeb<>nil
    if PDeb^=' ' then repeat inc(PDeb) until PDeb^<>' ';   // trim left ' '
    if IdemPChar(PDeb,UpperName) then
      exit;
  end;
  result := false;
end;

function GetSectionContent(SectionFirstLine: PUTF8Char): RawUTF8;
var PDeb: PUTF8Char;
begin
  PDeb := SectionFirstLine;
  while (SectionFirstLine<>nil) and (SectionFirstLine^<>'[') do
    GetNextLineBegin(SectionFirstLine,SectionFirstLine);
  if SectionFirstLine=nil then
    result := PDeb else
    SetString(result,PDeb,SectionFirstLine-PDeb);
end;

function GetSectionContent(const Content, SectionName: RawUTF8): RawUTF8; overload;
var P: PUTF8Char;
    UpperSection: array[byte] of AnsiChar;
begin
  P := pointer(Content);
  PWord(UpperCopy255(UpperSection,SectionName))^ := ord(']');
  if FindSectionFirstLine(P,UpperSection) then
    result := GetSectionContent(P) else
    result := '';
end;

function DeleteSection(var Content: RawUTF8; const SectionName: RawUTF8;
  EraseSectionHeader: boolean=true): boolean;
var P: PUTF8Char;
    UpperSection: array[byte] of AnsiChar;
begin
  result := false; // no modification
  P := pointer(Content);
  PWord(UpperCopy255(UpperSection,SectionName))^ := ord(']');
  if FindSectionFirstLine(P,UpperSection) then
    result := DeleteSection(P,Content,EraseSectionHeader);
end;

function DeleteSection(SectionFirstLine: PUTF8Char; var Content: RawUTF8;
  EraseSectionHeader: boolean=true): boolean;
var PEnd: PUTF8Char;
    IndexBegin: PtrInt;
begin
  result := false;
  PEnd := SectionFirstLine;
  if EraseSectionHeader then // erase [Section] header line
    while (PtrUInt(SectionFirstLine)>PtrUInt(Content)) and (SectionFirstLine^<>'[') do dec(SectionFirstLine);
  while (PEnd<>nil) and (PEnd^<>'[') do
    GetNextLineBegin(PEnd,PEnd);
  IndexBegin := SectionFirstLine-pointer(Content);
  if IndexBegin=0 then
    exit; // no modification
  if PEnd=nil then
    SetLength(Content,IndexBegin) else
    delete(Content,IndexBegin+1,PEnd-SectionFirstLine);
  result := true; // Content was modified
end;

procedure ReplaceSection(SectionFirstLine: PUTF8Char;
  var Content: RawUTF8; const NewSectionContent: RawUTF8); overload;
var PEnd: PUTF8Char;
    IndexBegin: PtrInt;
begin
  if SectionFirstLine=nil then
    exit;
  // delete existing [Section] content
  PEnd := SectionFirstLine;
  while (PEnd<>nil) and (PEnd^<>'[') do
    GetNextLineBegin(PEnd,PEnd);
  IndexBegin := SectionFirstLine-pointer(Content);
  if PEnd=nil then
    SetLength(Content,IndexBegin) else
    delete(Content,IndexBegin+1,PEnd-SectionFirstLine);
  // insert section content
  insert(NewSectionContent,Content,IndexBegin+1);
end;

procedure ReplaceSection(var Content: RawUTF8; const SectionName,
  NewSectionContent: RawUTF8);
var UpperSection: array[byte] of AnsiChar;
    P: PUTF8Char;
begin
  P := pointer(Content);
  PWord(UpperCopy255(UpperSection,SectionName))^ := ord(']');
  if FindSectionFirstLine(P,UpperSection) then
    ReplaceSection(P,Content,NewSectionContent) else
    Content := Content+'['+SectionName+']'#13#10+NewSectionContent;
end;

function FindIniNameValueInteger(P: PUTF8Char; UpperName: PUTF8Char): integer;
begin
  result := GetInteger(pointer(FindIniNameValue(P,UpperName)));
end;

function FindIniEntry(const Content, Section, Name: RawUTF8): RawUTF8;
var P: PUTF8Char;
    UpperSection, UpperName: array[byte] of AnsiChar;
    // possible GPF if length(Section/Name)>255, but should const in code
begin
  result := '';
  P := pointer(Content);
  if P=nil then exit;
  // UpperName := UpperCase(Name)+'=';
  PWord(UpperCopy255(UpperName,Name))^ := ord('=');
  if Section='' then
     // find the Name= entry before any [Section]
    result := FindIniNameValue(P,UpperName) else begin
     // find the Name= entry in the specified [Section]
    PWord(UpperCopy255(UpperSection,Section))^ := ord(']');
    if FindSectionFirstLine(P,UpperSection) then
      result := FindIniNameValue(P,UpperName);
  end;
end;

function FindWinAnsiIniEntry(const Content, Section,Name: RawUTF8): RawUTF8;
begin
  result := WinAnsiToUtf8(WinAnsiString(FindIniEntry(Content,Section,Name)));
end;

function FindIniEntryInteger(const Content,Section,Name: RawUTF8): integer;
begin
  result := GetInteger(pointer(FindIniEntry(Content,Section,Name)));
end;

function FindIniEntryFile(const FileName: TFileName; const Section,Name: RawUTF8): RawUTF8;
var Content: RawUTF8;
begin
  Content := StringFromFile(FileName);
  if Content='' then
    result := '' else
    result := FindIniEntry(Content,Section,Name);
end;

procedure UpdateIniEntry(var Content: RawUTF8; const Section,Name,Value: RawUTF8);
const CRLF = #13#10;
var P: PUTF8Char;
    PDeb: PUTF8Char;
    SectionFound: boolean;
    i, UpperNameLength: PtrInt;
    V: RawUTF8;
    UpperSection, UpperName: array[byte] of AnsiChar;
    // possible GPF if length(Section/Name)>255, but should be short const in code
label Sec;
begin
  PWord(UpperCopy255(UpperName,Name))^ := ord('=');
  UpperNameLength := length(Name)+1;
  V := Value+CRLF;
  P := pointer(Content);
  // 1. find Section, and try update within it
  if Section='' then
    goto Sec; // find the Name= entry before any [Section]
  SectionFound := false;
  PWord(UpperCopy255(UpperSection,Section))^ := ord(']');
  if FindSectionFirstLine(P,UpperSection) then begin 
Sec:  SectionFound := true;
      while (P<>nil) and (P^<>'[') do begin
        PDeb := GetNextLineBegin(P,P); // since PDeb=P, we have PDeb<>nil
        while PDeb^=' ' do inc(PDeb);   // trim left ' '
        if IdemPChar(PDeb,UpperName) then begin
          // update Name=Value entry
          inc(PDeb,UpperNameLength);
          i := (PDeb-pointer(Content))+1;
          if (i=length(Value)) and CompareMem(PDeb,pointer(Value),i) then
            exit; // new Value is identical to the old one -> no change
          if P=nil then // avoid last line (P-PDeb) calculation error
            SetLength(Content,i-1) else
            delete(Content,i,P-PDeb); // delete old Value
          insert(V,Content,i); // set new value
          exit;
        end;
      end;
      // we reached next [Section] without having found Name=
   end;
  // 2. section or Name= entry not found: add Name=Value
  V := Name+'='+V;
  if not SectionFound then
    // create not existing [Section]
    V := '['+Section+(']'+CRLF)+V;
  // insert Name=Value at P^ (end of file or end of [Section])
  if P=nil then
    // insert at end of file
    Content := Content+V else begin
    // insert at end of [Section]
    i := (P-pointer(Content))+1;
    insert(V,Content,i);
  end;
end;

procedure UpdateIniEntryFile(const FileName: TFileName; const Section,Name,Value: RawUTF8);
var Content: RawUTF8;
begin
  Content := StringFromFile(FileName);
  UpdateIniEntry(Content,Section,Name,Value);
  FileFromString(Content,FileName);
end;


function StringFromFile(const FileName: TFileName): RawByteString;
var F: THandle;
    Size: integer;
begin
  result := '';
  if FileName='' then
    exit;
  F := FileOpen(FileName,fmOpenRead or fmShareDenyNone);
  if F<>THandle(-1) then begin
    Size := GetFileSize(F,nil);
    SetLength(result,Size);
    if FileRead(F,pointer(Result)^,Size)<>Size then
      result := '';
    FileClose(F);
  end;
end;

function FileFromString(const Content: RawByteString; const FileName: TFileName;
  FlushOnDisk: boolean=false): boolean;
var F: THandle;
    L: integer;
begin
  result := false;
  F := FileCreate(FileName);
  if F=THandle(-1) then
    exit;
  if pointer(Content)<>nil then
    L := FileWrite(F,pointer(Content)^,length(Content)) else
    L := 0;
  result := (L=length(Content));
{$ifdef MSWINDOWS}
  if FlushOnDisk then
    FlushFileBuffers(F);
{$endif}
  FileClose(F);
end;

function GetFileNameWithoutExt(const FileName: TFileName): TFileName;
var i, max: PtrInt;
begin
  i := length(FileName);
  max := i-4;
  while (i>0) and not(cardinal(FileName[i]) in [ord('\'),ord('/'),ord('.')])
    and (i>=max) do dec(i);
  if (i=0) or (FileName[i]<>'.') then
    result := FileName else
    SetString(result,PChar(pointer(FileName)),i-1);
end;

function FileAgeToDateTime(const FileName: TFileName): TDateTime;
{$ifdef HASNEWFILEAGE}
begin
  if not FileAge(FileName,result) then
{$else}
var Age: integer;
begin
  Age := FileAge(FileName);
  if Age<>-1 then
    result := FileDateToDateTime(Age) else
{$endif}
    result := 0;
end;

{$IFDEF PUREPASCAL}
{$IFDEF UNICODE}
function Trim(const S: RawUTF8): RawUTF8;
var I, L: Integer;
begin
  L := Length(S);
  I := 1;
  while (I <= L) and (S[I] <= ' ') do Inc(I);
  if I > L then
    Result := '' else begin
    while S[L] <= ' ' do Dec(L);
    Result := Copy(S, I, L-I+1);
  end;
end;

function Pos(const substr, str: RawUTF8): Integer; overload;
begin // the RawByteString version is fast enough
  Result := System.Pos(RawByteString(substr),RawByteString(str));
end;
{$ENDIF}
{$ENDIF}

/// retrieve a property value in a text-encoded class
// - follows the Delphi serialized text object format, not standard .ini
// - if the property is a string, the simple quotes ' are trimed
function FindObjectEntry(const Content, Name: RawUTF8): RawUTF8;
var L: integer;
begin
  result := Trim(FindIniEntry(Content,'',Name+' ')); // 'Name = Value' format
  if (result<>'') and (result[1]='''') then begin
    L := length(result);
    if result[L]='''' then
      result := copy(result,2,L-2); // 'testDI6322.IAS' -> testDI6322.IAS
  end;
end;

/// retrieve a filename property value in a text-encoded class
// - follows the Delphi serialized text object format, not standard .ini
// - if the property is a string, the simple quotes ' are trimed
// - any file path and any extension are trimmed
function FindObjectEntryWithoutExt(const Content, Name: RawUTF8): RawUTF8;
begin
  result := RawUTF8(GetFileNameWithoutExt(
    ExtractFileName(TFileName(FindObjectEntry(Content,Name)))));
end;

function IntegerScanExists(P: PCardinalArray; Count: PtrInt; Value: cardinal): boolean;
{$ifdef PUREPASCAL}
var i: PtrInt; // very optimized code for speed
begin
  if P<>nil then begin
    result := true;
    for i := 1 to (Count shr 2) do   // 4 DWORD by loop - aligned read
      if (P^[0]=Value) or (P^[1]=Value) or
         (P^[2]=Value) or (P^[3]=Value) then
        exit else
        inc(PtrUInt(P),16);
    for i := 0 to (Count and 3)-1 do // last 0..3 DWORD
      if P^[i]=Value then
        exit;
  end;
  result := false;
end;
{$else}
asm // eax=P, edx=Count, Value=ecx
  test eax,eax
  jz @end // avoid GPF
  cmp edx,8
  jae @s1
  jmp dword ptr [edx*4+@Table]
  nop // align @Table
@Table:
  dd @z, @1, @2, @3, @4, @5, @6, @7
@s1: // fast search by 8 integers (pipelined instructions)
  sub edx,8
  cmp [eax],ecx;    je @ok
  cmp [eax+4],ecx;  je @ok
  cmp [eax+8],ecx;  je @ok
  cmp [eax+12],ecx; je @ok
  cmp [eax+16],ecx; je @ok
  cmp [eax+20],ecx; je @ok
  cmp [eax+24],ecx; je @ok
  cmp [eax+28],ecx; je @ok
  cmp edx,8
  lea eax,[eax+32] // preserve flags during 'cmp edx,8' computation
@s2:
  jae @s1
  jmp dword ptr [edx*4+@Table]
@7: cmp [eax+24],ecx; je @ok
@6: cmp [eax+20],ecx; je @ok
@5: cmp [eax+16],ecx; je @ok
@4: cmp [eax+12],ecx; je @ok
@3: cmp [eax+8],ecx;  je @ok
@2: cmp [eax+4],ecx;  je @ok
@1: cmp [eax],ecx;    je @ok
@z:
  xor eax,eax
@end:
  ret
@ok:
  mov al,1
end;
{$endif}

function IntegerScan(P: PCardinalArray; Count: PtrInt; Value: cardinal): PCardinal;
{$ifdef PUREPASCAL}
var i: PtrInt;
begin // very optimized code
  if P<>nil then begin
    for i := 1 to Count shr 2 do      // 4 DWORD by loop - aligned read
      if P^[0]<>Value then
      if P^[1]<>Value then
      if P^[2]<>Value then
      if P^[3]=Value then begin
        result := @P^[3];
        exit;
      end else
        inc(PtrUInt(P),16) else begin
        result := @P^[2];
        exit;
      end else begin
        result := @P^[1];
        exit;
      end else begin
        result := pointer(P);
        exit;
      end;
    for i := 0 to (Count and 3)-1 do  // last 0..3 DWORD
      if P^[i]=Value then begin
        result := @P^[i];
        exit;
      end;
  end;
  result := nil;
end;
{$else}
asm // eax=P, edx=Count, Value=ecx
       or eax,eax
       jz @ok0 // avoid GPF
       cmp edx,8
       jmp @s2
@ok0:  rep ret
@ok20: lea eax,eax+20; ret
@ok16: lea eax,eax+16; ret
@ok12: lea eax,eax+12; ret
@ok8:  lea eax,eax+8; ret
@ok4:  lea eax,eax+4; ret
@ok28: lea eax,eax+28; ret
@ok24: lea eax,eax+24; ret
       nop // align
@s1:   sub edx,8
       cmp [eax],ecx;    je @ok0
       cmp [eax+4],ecx;  je @ok4
       cmp [eax+8],ecx;  je @ok8
       cmp [eax+12],ecx; je @ok12
       cmp [eax+16],ecx; je @ok16
       cmp [eax+20],ecx; je @ok20
       cmp [eax+24],ecx; je @ok24
       cmp [eax+28],ecx; je @ok28
       cmp edx,8
       lea eax,[eax+32]  // preserve flags during 'cmp edx,8' computation
@s2:   jae @s1
       test edx,edx; jz @z
       cmp [eax],ecx;    je @ok0;  dec edx; jz @z
       cmp [eax+4],ecx;  je @ok4;  dec edx; jz @z
       cmp [eax+8],ecx;  je @ok8;  dec edx; jz @z
       cmp [eax+12],ecx; je @ok12; dec edx; jz @z
       cmp [eax+16],ecx; je @ok16; dec edx; jz @z
       cmp [eax+20],ecx; je @ok20; dec edx; jz @z
       cmp [eax+24],ecx; je @ok24
@z:    xor eax,eax // return nil if not found
end;
{$endif}

function AddInteger(var Values: TIntegerDynArray; Value: integer;
  NoDupplicates: boolean=false): boolean;
var n: PtrInt;
begin
  n := Length(Values);
  if NoDupplicates and IntegerScanExists(pointer(Values),n,Value) then begin
    result := false;
    exit;
  end;
  SetLength(Values,n+1);
  Values[n] := Value;
  result := true
end;

procedure DeleteInteger(var Values: TIntegerDynArray; Index: PtrInt);
var n: PtrInt;
begin
  n := Length(Values);
  if PtrUInt(Index)>=PtrUInt(n) then
    exit; // wrong Index
  dec(n);
  if n>Index then
    move(Values[Index+1],Values[Index],(n-Index)*sizeof(Integer));
  SetLength(Values,n);
end;

procedure CSVToIntegerDynArray(CSV: PUTF8Char; var Result: TIntegerDynArray);
begin
  while CSV<>nil do begin
    SetLength(Result,length(Result)+1);
    Result[high(Result)] := GetNextItemCardinal(CSV);
  end;
end;

function IntegerDynArrayToCSV(var Values: TIntegerDynArray): RawUTF8;
var i: PtrInt;
begin
  result := '';
  for i := 0 to high(Values) do
    result := result+{$ifndef ENHANCEDRTL}Int32ToUtf8{$else}IntToStr{$endif}(Values[i])+',';
  if result<>'' then
    SetLength(result,length(result)-1);
end;

function IntegerScanIndex(P: PCardinalArray; Count: PtrInt; Value: cardinal): PtrInt;
{$ifdef PUREPASCAL}
var i: PtrInt; // very optimized code for speed
begin
  if P<>nil then begin
    result := 0;
    for i := 1 to Count shr 2 do // 4 DWORD by loop - aligned read
      if P^[0]<>Value then
      if P^[1]<>Value then
      if P^[2]<>Value then
      if P^[3]<>Value then begin
        inc(PtrUInt(P),16);
        inc(result,4);
      end else begin
        inc(result,3);
        exit;
      end else begin
        inc(result,2);
        exit;
      end else begin
        inc(result,1);
        exit;
      end else
        exit;
    for i := 0 to (Count and 3)-1 do // last 0..3 DWORD
      if P^[i]=Value then
        exit else 
        inc(result);
  end;
  result := -1;
end;
{$else}
asm
    push eax
    call IntegerScan
    or eax,eax
    pop edx
    jz @z
    sub eax,edx
    shr eax,2
    ret
@z: mov eax,-1
end;
{$endif}

procedure QuickSortInteger(ID: PIntegerArray; L, R: PtrInt);
var I, J, P: PtrInt;
    pivot, Tmp: integer;
begin
  if L<R then
  repeat
    I := L; J := R;
    P := (L + R) shr 1;
    repeat
      pivot := ID^[P];
      while ID[I]<pivot do Inc(I);
      while ID[J]>pivot do Dec(J);
      if I <= J then begin
        Tmp := ID[J]; ID[J] := ID[I]; ID[I] := Tmp;
        if P = I then P := J else if P = J then P := I;
        Inc(I); Dec(J);
      end;
    until I > J;
    if L < J then
      QuickSortInteger(ID, L, J);
    L := I;
  until I >= R;
end;

function FastFindIntegerSorted(P: PIntegerArray; R: PtrInt; Value: integer): PtrInt;
var L: PtrInt;
    cmp: integer;
begin
  L := 0;
  if 0<=R then
  repeat
    result := (L + R) shr 1;
    cmp := P^[result]-Value;
    if cmp=0 then
      exit;
    if cmp<0 then
      L := result + 1 else
      R := result - 1;
  until (L > R);
  result := -1
end;

function GetInteger(P: PUTF8Char): PtrInt;
var c: PtrUInt;
    minus: boolean;
begin
  if P=nil then begin
    result := 0;
    exit;
  end;
  if P^=' ' then repeat inc(P) until P^<>' ';
  if P^='-' then begin
    minus := true;
    repeat inc(P) until P^<>' ';
  end else begin
    minus := false;
    if P^='+' then
      repeat inc(P) until P^<>' ';
  end;
  c := byte(P^)-48;
  if c>9 then
    result := 0 else begin
    result := c;
    inc(P);
    repeat
      c := byte(P^)-48;
      if c>9 then
        break else
        result := result*10+PtrInt(c);
      inc(P);
    until false;
  end;
  if minus then
    result := -result;
end;

function GetInteger(P: PUTF8Char; var err: integer): PtrInt;
var c: PtrUInt;
    minus: boolean;
begin
  if P=nil then begin
    result := 0;
    err := 1;
    exit;
  end else
    err := 0;
  if P^=' ' then repeat inc(P) until P^<>' ';
  if P^='-' then begin
    minus := true;
    repeat inc(P) until P^<>' ';
  end else begin
    minus := false;
    if P^='+' then
      repeat inc(P) until P^<>' ';
  end;
  c := byte(P^)-48;
  if c>9 then begin
    err := 1;
    result := 0;
    exit;
  end else begin
    result := c;
    inc(P);
    repeat
      c := byte(P^)-48;
      if c>9 then begin
        if byte(P^)<>0 then
          err := 1; // always return 1 as err code -> don't care about char index
        break;
      end else
        result := result*10+PtrInt(c);
      inc(P);
    until false;
  end;
  if minus then
    result := -result;
end;

function GetCardinal(P: PUTF8Char): PtrUInt;
var c: PtrUInt;
begin
  if P=nil then begin
    result := 0;
    exit;
  end;
  if P^=' ' then repeat inc(P) until P^<>' ';
  c := byte(P^)-48;
  if c>9 then
    result := 0 else begin
    result := c;
    inc(P);
    repeat
      c := byte(P^)-48;
      if c>9 then
        break else
        result := result*10+PtrUInt(c);
      inc(P);
    until false;
  end;
end;

function GetCardinalW(P: PWideChar): PtrUInt;
var c: PtrUInt;
begin
  if P=nil then begin
    result := 0;
    exit;
  end;
  if P^=' ' then repeat inc(P) until P^<>' ';
  c := word(P^)-48;
  if c>9 then
    result := 0 else begin
    result := c;
    inc(P);
    repeat
      c := word(P^)-48;
      if c>9 then
        break else
        result := result*10+c;
      inc(P);
    until false;
  end;
end;

{$ifdef CPU64}
function GetInt64(P: PUTF8Char): Int64;
begin // PtrInt is already int64 -> call previous version
  result := GetInt(P);
end;
{$else}
function GetInt64(P: PUTF8Char): Int64;
var c: cardinal;
    minus: boolean;
begin
  result := 0;
  if P=nil then
    exit;
  if P^=' ' then repeat inc(P) until P^<>' ';
  if P^='-' then begin
    minus := true;
    repeat inc(P) until P^<>' ';
  end else begin
    minus := false;
    if P^='+' then
      repeat inc(P) until P^<>' ';
  end;
  c := byte(P^)-48;
  if c>9 then
    exit;
  Int64Rec(result).Lo := c;
  inc(P);
  repeat
    c := byte(P^)-48;
    if c>9 then
      break else
      result := result shl 3+result+result; // fast result := result*10
      inc(result,c);
    inc(P);
  until false;
  if minus then
    result := -result;
end;
{$endif}

function GetInt64(P: PUTF8Char; var err: integer): Int64; overload;
{$ifdef ENHANCEDRTL}
begin
  val(PAnsiChar(P),result,err);
end;
{$else}
{$ifdef PUREPASCAL}
var c: cardinal;
    minus: boolean;
begin
  err := 0;
  result := 0;
  if P=nil then
    exit;
  if P^=' ' then repeat inc(P) until P^<>' ';
  if P^='-' then begin
    minus := true;
    repeat inc(P) until P^<>' ';
  end else begin
    minus := false;
    if P^='+' then
      repeat inc(P) until P^<>' ';
  end;
  Inc(err);
  c := byte(P^)-48;
  if c>9 then
    exit;
  Int64Rec(result).Lo := c;
  inc(P);
  repeat
    inc(err);
    if Byte(P^)=0 then begin
      err := 0;
      Break;
    end;
    c := byte(P^)-48;
    if c>9 then
      break else
      result := result shl 3+result+result; // fast result := result*10
      inc(result,c);
    inc(P);
  until false;
  if minus then
    result := -result;
end;
{$else}
asm // enhanced John O'Harrow code
  test  eax,eax
  jz    @@Null
  push  ebx
  push  esi
  push  edi
  push  edx                 {Save Code Address}
  push  eax                 {Save String Pointer}
  mov   esi,eax             {String Pointer}
  xor   ebx,ebx             {Clear Valid Flag and Sign Flag}
  xor   eax,eax             {Clear Result}
  xor   edx,edx
  jmp   @@TrimEntry
@@Null:
  mov   [edx],eax
  inc   [edx]               {Code = 1}
  xor   edx,edx             {Result = 0}
  ret
@@Trim:                     {Strip Leading Spaces}
  inc   esi
@@TrimEntry:
  movzx ecx,[esi]
  cmp   cl,' '
  je    @@Trim
  cmp   cl,'0'
  jle   @@CheckFirstChar
@@CheckAlpha:
  test  cl,$87
  jz    @@CheckX            {May be 'x' or 'X'}
@@NumLoop:
  sub   ecx,'0'
  cmp   ecx,9
  ja    @@NumDone           {Not '0'..'9'}
  cmp   eax,MaxInt/10-9     {(MaxInt div 10)-9}
  ja    @@LargeNum
  lea   eax,[eax*4+eax]
  lea   eax,[eax*2+ecx]     {Result = Result * 10 + Digit}
  inc   esi
  mov   bl,1                {Valid := True}
  movzx ecx,[esi]
  jmp   @@NumLoop
@@LargeNum:
  mov   bh,cl               {Save Digit}
  add   eax,eax
  adc   edx,edx
  mov   ecx,eax
  mov   edi,edx             {edi:ecx = Result * 2}
  shld  edx,eax,2
  add   eax,eax
  add   eax,eax             {edx:eax = Result * 8}
  add   eax,ecx
  adc   edx,edi             {Result = Result * 10}
  movzx ecx,bh              {Restore Digit}
  add   eax,ecx             {Add Digit to Result}
  adc   edx,0
  inc   esi
  movzx ecx,[esi]
  sub   ecx,'0'
  cmp   ecx,9
  ja    @@NumDone           {Not '0'..'9'}
  cmp   edx,$0ccccccc       {May be Out of Range?}
  jb    @@LargeNum
  ja    @@SetSign           {Out of Range}
  cmp   eax,$cccccccc
  jna   @@LargeNum          {Within Range}
  jmp   @@SetSign
@@NumDone:
  cmp   edx,$80000000       {Check for Overflow}
  jb    @@SetSign
  jne   @@Overflow
  test  eax,eax
  jnz   @@Overflow
  test  ebx,ebx             {Sign Flag}
  js    @@Setsign           {Result is Valid (-MaxInt64-1)}
@@Overflow:
  dec   esi
  mov   bl,0               {Valid := False}
  jmp   @@SetSign
@@CheckFirstChar:
  cmp   cl,'-'
  je    @@PlusMinus
  cmp   cl,'+'
  jne   @@SignSet
@@PlusMinus:                {Starts with '+' or '-'}
  mov   bl,'+'+1
  sub   ebx,ecx             {Set Sign Flag: '+' -> +1, '-' -> -1}
  inc   esi
  mov   bl,0                {Valid := False}
  movzx ecx,[esi]           {Character after '+' or '-'}
@@SignSet:
  cmp   cl,'$'
  je    @@Hex               {Hexadecimal}
  cmp   cl,'0'
  jne   @@CheckAlpha        {May start with 'x' or 'X'}
  inc   esi
  mov   bl,1                {Assume Valid = True}
  movzx ecx,[esi]           {Character after '0'}
  jmp   @@CheckAlpha        {May start with '0x' or '0X'}
@@CheckX:
  mov   bh,cl
  or    bh,$20              {'X' -> 'x'}
  cmp   bh,'x'
  jne   @@NumLoop
@@Hex:
  mov   bl,0                {Valid := False}
@@HexLoop:
  inc   esi
  movzx ecx,[esi]
  cmp   cl,'a'
  jb    @@CheckNum
  sub   cl,'a'-'A'          {'a' > 'A'}
@@CheckNum:
  sub   cl,'0'
  cmp   cl,9
  jna   @@CheckHexRange     {'0'..'9'}
  sub   cl,'A'-'0'
  cmp   cl,5                {Valid Hex Character?}
  ja    @@NotHex            {No: Invalid}
  add   cl,10               {Yes: Adjust Digit}
@@CheckHexRange:
  cmp   edx,$10000000
  jae   @@SetSign          {Overflow}
  shld  edx,eax,4          {Result := Result * 16}
  shl   eax,4
  add   eax,ecx            {Add Digit}
  adc   edx,0
  mov   bl,1               {Valid := True}
  jmp   @@HexLoop
@@NotHex:
  add   cl,'A'-'0'         {Restore Char-'0'}
@@SetSign:
  mov   ch,bl              {Save Valid Flag}
  sar   ebx,31             {Set Each Bit to Top Bit (Sign Flag)}
  xor   eax,ebx            {Negate Result if Necessary}
  xor   edx,ebx
  sub   eax,ebx
  sbb   edx,ebx
  dec   ch                  {0 if Valid,-1 if Invalid}
  or    cl,ch               {If Invalid, Force CL = -1}
  cmp   cl,-'0'
  jne   @@Error             {Not Valid or Not End of String}
  xor   esi,esi             {Code := 0}
  pop   ebx                 {Dump String Pointer}
@@Finished:
  pop   ecx
  mov   [ecx],esi           {Set Error Code}
  pop   edi
  pop   esi
  pop   ebx
  ret
@@Error:
  inc   esi
  pop   ecx                 {String Pointer}
  sub   esi,ecx
  jmp   @@Finished
end;
{$endif}
{$endif}

function GetExtended(P: PUTF8Char; var err: integer): extended;
{$ifdef ENHANCEDRTL}
begin
  val(PAnsiChar(P),result,err);
end;
{$else}
{$ifdef PUREPASCAL}
begin
  val(string(P),result,err);
end;
{$else}
const
  Ten: Double = 10.0;
// faster ValExt_JOH_IA32_8_a implementation by John O'Harrow
// also avoid val() conversion into UnicodeString for Delphi 2009/2010
asm   // -> EAX Pointer to string
      //    EDX Pointer to code result
      // <- FST(0)  Result
  push  ebx              {Save Used Registers}
  push  esi
  push  edi
  mov   esi,eax         {String Pointer}
  push  eax             {Save for Error Condition}
  xor   ebx,ebx
  push  eax             {Allocate Local Storage for Loading FPU}
  test  esi,esi
  jz    @@Nil           {Nil String}
@@Trim:
  movzx ebx,[esi]       {Strip Leading Spaces}
  inc   esi
  cmp   bl,' '
  je    @@Trim
  xor   ecx,ecx         {Clear Sign Flag}
{$IFDEF PIC}
  call  GetGOT
  fld   qword [eax.Ten] {Load 10 into FPU} {$ELSE}
  fld   qword [Ten]     {Load 10 into FPU}
{$ENDIF}
  xor   eax,eax         {Zero Number of Decimal Places}
  fldz                  {Zero Result in FPU}
  cmp   bl,'0'
  jl    @@CheckSign     {Check for Sign Character}
@@FirstDigit:
  xor   edi,edi         {Zero Exponent Value}
@@DigitLoop:
  sub   bl,'0'
  cmp   bl, 9
  ja    @@Fraction      {Non-Digit}
  mov   cl,1            {Set Digit Found Flag}
  mov   [esp],ebx       {Store for FPU Use}
  fmul  st(0), st(1)    {Multply by 10}
  fiadd dword ptr [esp] {Add Next Digit}
  movzx ebx,[esi]       {Get Next Char}
  inc   esi
  test  bl,bl           {End Reached?}
  jnz   @@DigitLoop     {No, Get Next Digit}
  jmp   @@Finish        {Yes, Finished}
@@CheckSign:
  cmp   bl,'-'
  je    @@Minus
  cmp   bl,'+'
  je    @@SignSet
@@GetFirstDigit:
  test  bl, bl
  jz    @@Error         {No Digits Found}
  jmp   @@FirstDigit
@@Minus:
  mov   ch,1            {Set Sign Flag}
@@SignSet:
  movzx ebx,[esi]       {Get Next Char}
  inc   esi
  jmp   @@GetFirstDigit
@@Fraction:
  cmp   bl,'.'-'0'
  jne   @@Exponent      {No Decimal Point}
  movzx ebx,[esi]       {Get Next Char}
  test  bl, bl
  jz    @@DotEnd        {String Ends with '.'}
  inc   esi
@@FractionLoop:
  sub   bl,'0'
  cmp   bl, 9
  ja    @@Exponent       {Non-Digit}
  mov   [esp],ebx
  dec   eax              {-(Number of Decimal Places)}
  fmul  st(0), st(1)     {Multply by 10}
  fiadd dword ptr [esp]  {Add Next Digit}
  movzx ebx,[esi]        {Get Next Char}
  inc   esi
  test  bl,bl            {End Reached?}
  jnz   @@FractionLoop   {No, Get Next Digit}
  jmp   @@Finish         {Yes, Finished (No Exponent)}
@@DotEnd:
  test  cl,cl            {Any Digits Found before '.'?}
  jnz   @@Finish         {Yes, Valid}
  jmp   @@Error          {No, Invalid}
@@Exponent:
  or    bl, $20
  cmp   bl,'e'-'0'
  jne   @@Error          {Not 'e' or 'E'}
@@GetExponent:
  movzx ebx,[esi]        {Get Next Char}
  inc   esi
  mov   cl, 0            {Clear Exponent Sign Flag}
  cmp   bl,'-'
  je    @@MinusExp
  cmp   bl,'+'
  je    @@ExpSignSet
  jmp   @@ExpLoop
@@MinusExp:
  mov   cl,1            {Set Exponent Sign Flag}
@@ExpSignSet:
  movzx ebx,[esi]       {Get Next Char}
  inc   esi
@@ExpLoop:
  sub   bl,'0'
  cmp   bl, 9
  ja    @@Error         {Non-Digit}
  lea   edi,[edi+edi*4] {Multiply by 10}
  add   edi,edi
  add   edi,ebx         {Add Next Digit}
  movzx ebx,[esi]       {Get Next Char}
  inc   esi
  test  bl, bl           {End Reached?}
  jnz   @@ExpLoop        {No, Get Next Digit}
@@EndExp:
  test  cl, cl           {Positive Exponent?}
  jz    @@Finish         {Yes, Keep Exponent Value}
  neg   edi              {No, Negate Exponent Value}
@@Finish:
  add   eax,edi         {Exponent Value - Number of Decimal Places}
  mov   [edx],ebx       {Result Code = 0}
  jz    @@PowerDone     {No call to _Pow10 Needed}
  mov   edi,ecx         {Save Decimal Sign Flag}
  call  System.@Pow10          {Raise to Power of 10}
  mov   ecx,edi         {Restore Decimal Sign Flag}
@@PowerDone:
  test  ch, ch           {Decimal Sign Flag Set?}
  jnz   @@Negate         {Yes, Negate Value}
@@Success:
  add   esp, 8           {Dump Local Storage and String Pointer}
@@Exit:
  ffree st(1)            {Remove Ten Value from FPU}
  pop   edi              {Restore Used Registers}
  pop   esi
  pop   ebx
  ret                    {Finished}
@@Negate:
  fchs                   {Negate Result in FPU}
  jmp   @@Success
@@Nil:
  inc   esi              {Force Result Code = 1}
  fldz                   {Result Value = 0}
@@Error:
  pop   ebx              {Dump Local Storage}
  pop   eax              {String Pointer}
  sub   esi,eax          {Error Offset}
  mov   [edx],esi        {Set Result Code}
  test  ch, ch           {Decimal Sign Flag Set?}
  jz    @@Exit           {No, exit}
  fchs                   {Yes. Negate Result in FPU}
  jmp   @@Exit           {Exit Setting Result Code}
end;
{$endif}
{$endif}

function GetUTF8Char(P: PUTF8Char): PtrUInt;
begin
  if P<>nil then begin
    result := byte(P[0]);
    if result and $80<>0 then
      if result and $20=0 then // UTF-8 decode
        result := result shl 6+byte(P[1])-$00003080 else
        result := (result shl 6+byte(P[1]))shl 6+byte(P[2])-$000E2080;
  end else
    result := PtrUInt(P);
end;

function NextUTF8Char(P: PUTF8Char; out Next: PUTF8Char): PtrUInt;
begin
  if P<>nil then begin
    result := byte(P[0]);
    inc(P);
    if result and $80=0 then
      Next := P else
      if result and $20=0  then begin // UTF-8 decode
        result := result shl 6+byte(P[0])-$00003080;
        Next := P+1;
      end else begin
        result := (result shl 6+byte(P[0]))shl 6+byte(P[1])-$000E2080;
        Next := P+2;
      end;
  end else begin
    result := PtrUInt(P);
    Next := P;
  end;
end;

function ContainsUTF8(p, up: PUTF8Char): boolean;
var u: PByte;
begin
  if (p<>nil) and (up<>nil) and (up^<>#0) then begin
    result := true;
    repeat
      u := pointer(up);
      repeat
        if GetNextUTF8Upper(p)<>u^ then
          break else
          inc(u);
        if u^=0 then
          exit; // up^ was found inside p^
      until false;
      p := FindNextUTF8WordBegin(p);
    until p=nil;
  end;
  result := false;
end;

function IdemFileExt(p, extup: PUTF8Char): Boolean;
var ext: PUTF8Char;
begin
  if (p<>nil) and (extup<>nil) then begin
    ext := nil;
    repeat
      if p^='.' then
        ext := p; // get last '.' position from p into ext
      inc(p);
    until p^=#0;
    if ext<>nil then
      result := IdemPChar(ext,extup) else
      result := false;
  end else
    result := false;
end;

function IdemPChar(p, up: PUTF8Char): boolean;
// if the beginning of p^ is same as up^ (ignore case - up^ must be already Upper)
{$ifdef USENORMTOUPPER}
{$ifdef PUREPASCAL}
begin
  result := false;
  if (p=nil) or (up=nil) then
    exit;
  while up^<>#0 do begin
    if up^<>NormToUpper[p^] then
      exit;
    inc(up);
    inc(p);
  end;
  result := true;
end;
{$else}
// eax=p edx=up
asm
  or eax,eax
  jz @e // P=nil -> false
  or edx,edx
  push ebx
  push esi
  jz @z // up=nil -> true
  mov esi,offset NormToUpper
  xor ebx,ebx
  xor ecx,ecx
@1:
  mov cl,[edx] // cl=up^
  mov bl,[eax] // bl=p^
  test cl,cl
  mov bl,[ebx+esi] // bl=NormToUpper[p^]
  jz @z // up^=#0 -> OK
  lea edx,[edx+1] // = inc edx without changing flags
  cmp bl,cl
  lea eax,[eax+1]
  je @1
  pop esi
  pop ebx
  xor eax,eax
@e:
  ret
@z:
  mov al,1 // up^=#0 -> OK
  pop esi
  pop ebx
end;
{$endif}
{$else}
var c: AnsiChar;
begin
  result := false;
  if (p=nil) or (up=nil) then
    exit;
  while up^<>#0 do begin
    c := p^;
    if up^<>c then
      if c in ['a'..'z'] then begin
        dec(c,32);
        if up^<>c then
          exit;
      end else exit;
    inc(up);
    inc(p);
  end;
  result := true;
end;
{$endif USENORMTOUPPER}

function UpperCase(const source: RawUTF8): RawUTF8;
begin
  result := '';
  SetLength(result,length(source));
{$ifdef UNICODE2} // not needed: SetLength() did already set the codepage
  PWord(PtrInt(result)-12)^ := CP_UTF8; // use only SetLength() -> force set code page
{$endif}
  UpperCopy(pointer(result),source);
end;

function UpperCopy255(dest: PAnsiChar; const source: RawUTF8): PAnsiChar;
{$ifdef USENORMTOUPPER}
{$ifdef PUREPASCAL}
var s: PAnsiChar;
    c: cardinal;
begin
  s := pointer(source);
  if s<>nil then
    repeat
      c := ord(s^);
      if c=0 then
        break else
        dest^ := AnsiChar(NormToUpperByte[c]);
      inc(s);
      inc(dest);
    until false;
  result := dest;
end;
{$else}
asm // eax=dest source=edx
    or edx,edx
    jz @z
    cmp [edx-4],250
    ja @z // avoid buffer overflow (expect dest: array[byte] of char)
    push esi
    mov esi,offset NormToUpper
    xor ecx,ecx
@1: mov cl,[edx]
    inc edx
    or cl,cl
    mov cl,[esi+ecx]
    jz @2
    mov [eax],cl
    inc eax
    jmp @1
@2: pop esi
@z:
end;
{$endif}
{$else}
var c: AnsiChar;
    s: PAnsiChar;
begin
  s := pointer(source);
  if s<>nil then
    repeat
      c := s^;
      if c=#0 then break;
      if c in ['a'..'z'] then
        dec(c,32);
      dest^ := c;
      inc(s);
      inc(dest);
    until false;
  result := dest;
end;
{$endif}

function UpperCopy(dest: PAnsiChar; const source: RawUTF8): PAnsiChar;
{$ifdef USENORMTOUPPER}
{$ifdef PUREPASCAL}
var s: PAnsiChar;
    c: cardinal;
begin
  s := pointer(source);
  if s<>nil then
    repeat
      c := ord(s^);
      if c=0 then
        break else
        dest^ := AnsiChar(NormToUpperByte[c]);
      inc(s);
      inc(dest);
    until false;
  result := dest;
end;
{$else}
asm // eax=dest source=edx
    or edx,edx
    jz @z
    push esi
    mov esi,offset NormToUpper
    xor ecx,ecx
@1: mov cl,[edx]
    inc edx
    or cl,cl
    mov cl,[esi+ecx]
    jz @2
    mov [eax],cl
    inc eax
    jmp @1
@2: pop esi
@z:
end;
{$endif}
{$else}
var c: AnsiChar;
    s: PAnsiChar;
begin
  s := pointer(source);
  if s<>nil then
    repeat
      c := s^;
      if c=#0 then break;
      if c in ['a'..'z'] then
        dec(c,32);
      dest^ := c;
      inc(s);
      inc(dest);
    until false;
  result := dest;
end;
{$endif}

function UpperCopyShort(dest: PAnsiChar; const source: shortstring): PAnsiChar;
{$ifdef USENORMTOUPPER}
{$ifdef PUREPASCAL}
var i: PtrInt;
begin
  for i := 1 to ord(source[0]) do begin
    dest^ := AnsiChar(NormToUpperByte[ord(source[i])]);
    inc(dest);
  end;
  result := dest;
end;
{$else}
asm // eax=dest source=edx
    push esi
    push ebx
    movzx ebx,byte ptr [edx] // ebx = length(source)
    xor ecx,ecx
    or ebx,ebx
    mov esi,offset NormToUpper
    jz @2 // source=''
    inc edx
@1: mov cl,[edx]
    inc edx
    dec ebx
    mov cl,[esi+ecx]
    mov [eax],cl
    lea eax,eax+1
    jnz @1
@2: pop ebx
    pop esi
@z:
end;
{$endif}
{$else}
var c: AnsiChar;
    i: PtrInt;
begin
  for i := 1 to ord(source[0]) do begin
    c := source[i];
    if c in ['a'..'z'] then
      dec(c,32);
    dest^ := c;
    inc(dest);
  end;
  result := dest;
end;
{$endif}

function GetNextLine(source: PUTF8Char; out next: PUTF8Char): RawUTF8;
begin
  next := source;
  if source=nil then begin
    result := '';
    exit;
  end;
  while not (source^ in [#0,#10,#13]) do inc(source);
  SetString(result,PAnsiChar(next),source-next);
  if source^=#13 then inc(source);
  if source^=#10 then inc(source);
  if source^=#0 then
    next := nil else
    next := source;
end;

{$ifdef UNICODE}
function GetNextLineW(source: PWideChar; out next: PWideChar): string;
begin
  next := source;
  if source=nil then begin
    result := '';
    exit;
  end;
  while not (cardinal(source^) in [0,10,13]) do inc(source);
  SetString(result,PChar(next),source-next);
  if source^=#13 then inc(source);
  if source^=#10 then inc(source);
  if source^=#0 then
    next := nil else
    next := source;
end;

function FindIniNameValueW(P: PWideChar; UpperName: PUTF8Char): string;
var PDeb: PWideChar;
    L: PtrInt;
begin
  while (P<>nil) and (P^<>'[') do begin
    PDeb := P;
    while not (cardinal(P^) in [0,10,13]) do inc(P);
    while cardinal(P^) in [10,13] do inc(P);
    if P^=#0 then P := nil;
    if PDeb^=' ' then repeat inc(PDeb) until PDeb^<>' ';   // trim left ' '
    if IdemPCharW(PDeb,UpperName) then begin
      inc(PDeb,StrLen(UpperName));
      L := 0; while PDeb[L]>=' ' do inc(L); // get line length
      SetString(result,PDeb,L);
      exit;
    end;
  end;
  result := '';
end;

function FindIniEntryW(const Content: string; const Section, Name: RawUTF8): string;
var P: PWideChar;
    UpperSection, UpperName: array[byte] of AnsiChar;
    // possible GPF if length(Section/Name)>255, but should const in code
begin
  result := '';
  P := pointer(Content);
  if P=nil then exit;
  // UpperName := UpperCase(Name)+'=';
  PWord(UpperCopy255(UpperName,Name))^ := ord('=');
  if Section='' then
     // find the Name= entry before any [Section]
    result := FindIniNameValueW(P,UpperName) else begin
     // find the Name= entry in the specified [Section]
    PWord(UpperCopy255(UpperSection,Section))^ := ord(']');
    if FindSectionFirstLineW(P,UpperSection) then
      result := FindIniNameValueW(P,UpperName);
  end;
end;

{$endif}


function IdemPCharAndGetNextLine(var source: PUTF8Char; search: PUTF8Char): boolean;
{$ifdef PUREPASCAL}
begin
  if source=nil then
    result := false else begin
    result := IdemPChar(source,search);
    while not (source^ in [#0,#10,#13]) do inc(source);
    while source^ in [#13,#10] do inc(source);
    if source^=#0 then
      source := nil;
  end;
end;
{$else}
asm // eax=source edx=search
    push eax       // save source var
    mov eax,[eax]  // eax=source
    or eax,eax
    jz @z
    push eax
    call IdemPChar
    pop ecx       // ecx=source
    push eax      // save result
@1: mov dl,[ecx]  // while not (source^ in [#0,#10,#13]) do inc(source);
    inc ecx
    cmp dl,13
    ja @1
    je @e
    or dl,dl
    jz @0
    cmp dl,10
    jne @1
    jmp @4
@e: cmp byte ptr [ecx],10 // jump #13#10
    jne @4
@3: inc ecx
@4: pop eax        // restore result
    pop edx        // restore source var
    mov [edx],ecx  // update source var
    ret
@0: xor ecx,ecx    // set source=nil
    jmp @4
@z: pop edx       // ignore source var, result := false
end;
{$endif}

function GetNextLineBegin(source: PUTF8Char; out next: PUTF8Char): PUTF8Char;
begin
  result := pointer(source);
  if source=nil then
    exit;
  while not (source^ in [#0,#10,#13]) do inc(source);
  if source^=#13 then inc(source);
  if source^=#10 then inc(source);
  if source^=#0 then
    next := nil else
    next := source;
end;

function GetNextItem(var P: PUTF8Char; Sep: AnsiChar= ','): RawUTF8;
var S: PUTF8Char;
begin
  if P=nil then
    result := '' else begin
    S := P;
    while (S^<>#0) and (S^<>Sep) do
      inc(S);
    SetString(result,P,S-P);
    if S^<>#0 then
     P := S+1 else
     P := nil;
  end;
end;

function GetNextItemString(var P: PChar; Sep: Char= ','): string;
// this function will compile into AnsiString or UnicodeString, depending
// of the compiler version
var S: PChar;
begin
  if P=nil then
    result := '' else begin
    S := P;
    while (S^<>#0) and (S^<>Sep) do
      inc(S);
    SetString(result,P,S-P);
    if S^<>#0 then
     P := S+1 else
     P := nil;
  end;
end;

procedure AppendCSVValues(const CSV: string; const Values: array of string;
  var Result: string; const AppendBefore: string=#13#10);
var Caption: string;
    i, bool: integer;
    P: PChar;
    first: Boolean;
begin
  P := pointer(CSV);
  if P=nil then
    exit;
  first := True;
  for i := 0 to high(Values) do begin
    Caption := GetNextItemString(P);
    if Values[i]<>'' then begin
      if first then begin
        Result := Result+#13#10;
        first := false;
      end else
        Result := Result+AppendBefore;
      bool := FindCSVIndex('0,-1',RawUTF8(Values[i]));
      Result := Result+Caption+': ';
      if bool<0 then
        Result := Result+Values[i] else
        Result := Result+GetCSVItemString(pointer(GetNextItemString(P)),bool,'/');
    end;
  end;
end;

procedure GetNextItemShortString(var P: PUTF8Char; out Dest: ShortString; Sep: AnsiChar= ',');
var S: PUTF8Char;
begin
  if P=nil then
    Dest[0] := #0 else begin
    S := P;
    while (S^<>#0) and (S^<>Sep) do
      inc(S);
    SetString(Dest,P,S-P);
    if S^<>#0 then
     P := S+1 else
     P := nil;
  end;
end;

function GetNextItemCardinal(var P: PUTF8Char; Sep: AnsiChar= ','): PtrUInt;
var c: PtrUInt;
begin
  if P=nil then begin
    result := 0;
    exit;
  end;
  c := byte(P^)-48;
  if c>9 then
    result := 0 else begin
    result := c;
    inc(P);
    repeat
      c := byte(P^)-48;
      if c>9 then
        break else
        result := result*10+c;
      inc(P);
    until false;
  end;
  while (P^<>#0) and (P^<>Sep) do // go to end of CSV item (ignore any decimal)
    inc(P);
  if P^=#0 then
    P := nil else
    inc(P);
end;

function GetNextItemCardinalW(var P: PWideChar; Sep: WideChar= ','): PtrUInt;
var c: PtrUInt;
begin
  if P=nil then begin
    result := 0;
    exit;
  end;
  c := word(P^)-48;
  if c>9 then
    result := 0 else begin
    result := c;
    inc(P);
    repeat
      c := word(P^)-48;
      if c>9 then
        break else
        result := result*10+c;
      inc(P);
    until false;
  end;
  while (P^<>#0) and (P^<>Sep) do // go to end of CSV item (ignore any decimal)
    inc(P);
  if P^=#0 then
    P := nil else
    inc(P);
end;

function GetNextItemDouble(var P: PUTF8Char; Sep: AnsiChar= ','): double;
var tmp: RawUTF8;
    err: integer;
begin
  if P=nil then
    result := 0 else begin
    tmp := GetNextItem(P,Sep);
    result := GetExtended(pointer(tmp),err);
    if err<>0 then
      result := 0;
  end;
end;

function GetCSVItem(P: PUTF8Char; Index: PtrUInt; Sep: AnsiChar = ','): RawUTF8;
var i: PtrUInt;
begin
  if P=nil then
    result := '' else
    for i := 0 to Index do
      result := GetNextItem(P,Sep);
end;

function GetCSVItemString(P: PChar; Index: PtrUInt; Sep: Char = ','): string;
var i: PtrUInt;
begin
  if P=nil then
    result := '' else
    for i := 0 to Index do
      result := GetNextItemString(P,Sep);
end;

function FindCSVIndex(CSV: PUTF8Char; const Value: RawUTF8; Sep: AnsiChar = ',';
  CaseSensitive: boolean=true): integer;
var s: RawUTF8;
begin
  result := 0;
  while CSV<>nil do begin
    s := GetNextItem(CSV);
    if CaseSensitive then begin
      if s=Value then
        exit;
    end else
     if SameTextU(s,Value) then
       exit;
    inc(result);
  end;
  result := -1; // not found
end;

procedure CSVToRawUTF8DynArray(CSV: PUTF8Char; var Result: TRawUTF8DynArray; Sep: AnsiChar = ',');
var s: RawUTF8;
begin
  while CSV<>nil do begin
    s := GetNextItem(CSV,Sep);
    if s<>'' then begin
      SetLength(Result,length(Result)+1);
      Result[high(Result)] := s;
    end;
  end;
end;

function UrlEncode(const svar: RawUTF8): RawUTF8;
function Enc(s, p: PUTF8Char): PUTF8Char;
var c: PtrInt;
begin
  repeat
    c := ord(s^);
    case c of
    0: break;
    ord('0')..ord('9'),ord('a')..ord('z'),ord('A')..ord('Z'),
    ord('_'),ord('-'),ord('.'),ord('~'): begin
      // cf. rfc3986 2.3. Unreserved Characters
      p^ := AnsiChar(c);
      inc(p);
      inc(s);
      continue;
    end;
    ord(' '): p^ := '+';
    else begin
      p^ := '%'; inc(p);
      p^ := HexChars[c shr 4]; inc(p);
      p^ := HexChars[c and $F];
    end;
    end; // case c of
    inc(p);
    inc(s);
  until false;
  result := p;
end;
function Size(s: PUTF8Char): PtrInt;
begin
  result := 0;
  if s<>nil then 
  repeat
    case s^ of
      #0: exit;
      '0'..'9','a'..'z','A'..'Z','_','-','.','~',' ': begin
        inc(result);
        inc(s);
        continue;
      end;
      else inc(result,3);
    end;
    inc(s);
  until false;
end;
begin
  result := '';
  if pointer(svar)=nil then
    exit else
    SetLength(result,Size(pointer(svar))); // reserve exact memory count
  Enc(pointer(svar),pointer(result));
end;

// ! url := UrlEncode(['select','*','where','ID=12','offset',23]);
function UrlEncode(const NameValuePairs: array of const): RawUTF8;
var A, n: PtrInt;
    name, value: RawUTF8;
function Invalid(P: PAnsiChar): boolean;
begin
  result := true;
  if P<>nil then begin
    repeat
      if not (P^ in ['a'..'z','A'..'Z']) then
        exit else
        inc(P);
    until P^=#0;
    result := false;
  end;
end;
begin
  result := '';
  n := high(NameValuePairs);
  if n>0 then begin
    for A := 0 to n shr 1 do begin
      VarRecToUTF8(NameValuePairs[A*2],name);
      if Invalid(pointer(name)) then
        continue;
      VarRecToUTF8(NameValuePairs[A*2+1],value);
      result := result+'&'+name+'='+UrlEncode(value);
    end;
    result[1] := '?';
  end;
end;

function UrlDecode(const s: RawUTF8; i: PtrInt = 1; len: PtrInt = -1): RawUTF8;
var L: PtrInt;
    P: PUTF8Char;
label SP;
begin
  result := '';
  if s='' then
    exit;
  L := PInteger(PtrInt(s)-4)^;
  if len<0 then
    len := L;
  if i>L then
    exit;
  dec(i);
  if len=i then
    exit;
  if Pointer(ConvertHexToBin)=nil then
    InitConvertHexToBin;
  Setlength(result,len-i); // reserve enough space for result
{$ifdef UNICODE2} // not needed: SetLength() did already set the codepage
  PWord(PtrInt(result)-12)^ := CP_UTF8; // use only SetLength() -> force set code page
{$endif}
  P := pointer(result);
  while i<len do begin
    case s[i+1] of
      #0: break; // reached end of s
      '%': if i+3>len then
        break else begin
        inc(i,2);
        P^ := AnsiChar(ConvertHexToBin[ord(s[i])] shl 4+ConvertHexToBin[ord(s[i+1])]);
      end;
      '+': P^  := ' ';
    else
      P^ := s[i+1];
    end; // case s[i] of
    inc(i);
    inc(P);
  end;
  Setlength(result,P-pointer(Result)); // very fast with FastMM4 (in-place realloc)
end;

function UrlDecodeValue(U, Upper: PUTF8Char; var Value: RawUTF8; Next: PPUTF8Char=nil): boolean;
var Beg, V: PUTF8Char;
    len, i: PtrInt;
begin
  // UrlDecodeValue('select=%2A&where=LastName%3D%27M%C3%B4net%27','SELECT=',V,@U)
  // -> U^='where=...' and V='*'
  result := false; // mark value not modified by default
  if U=nil then begin
    if Next<>nil then
      Next^ := U;
    exit;
  end;
  if IdemPChar(U,Upper) then begin
    result := true;
    if Pointer(ConvertHexToBin)=nil then
      InitConvertHexToBin;
    inc(U,StrLen(Upper));
    // compute resulting length of value
    Beg := U;
    len := 0;
    while not(U^ in [#0,'&']) do begin
      if U^='%' then
        if (U[1]=#0) or (U[2]=#0) then // avoid buffer overflow
          break else
          inc(U,3) else
        inc(U);
      inc(len);
    end;
    // decode value content
    SetLength(Value,len);
    V := pointer(Value);
    U := Beg;
    for i := 1 to len do
      if U^='%' then begin
        V^ := AnsiChar(ConvertHexToBin[ord(U[1])] shl 4+ConvertHexToBin[ord(U[2])]);
        inc(V);
        inc(U,3);
      end else begin
        if U^='+' then
          V^ := ' ' else
          V^ := U^;
        inc(V);
        inc(U);
      end;
  end;
  if Next=nil then
    exit;
  while not(U^ in [#0,'&']) do inc(U);
  if U^=#0 then
    Next^ := nil else
    Next^ := U+1; // jump '&'
end;

function UrlDecodeInteger(U, Upper: PUTF8Char; var Value: integer; Next: PPUTF8Char=nil): boolean;
var V: PtrInt;
    SignNeg: boolean;
begin
  // UrlDecodeInteger('offset=20&where=LastName%3D%27M%C3%B4net%27','OFFSET=',O,@Next)
  // -> Next^='where=...' and O=20
  result := false; // mark value not modified by default
  if U=nil then begin
    if Next<>nil then
      Next^ := U;
    exit;
  end;
  if IdemPChar(U,Upper) then begin
    inc(U,StrLen(Upper));
    if U^='-' then begin
      SignNeg := True;
      Inc(U);
    end else
      SignNeg := false;
    if U^ in ['0'..'9'] then begin
      V := 0;
      repeat
        V := (V*10)+ord(U^)-48;
        inc(U);
      until not (U^ in ['0'..'9']);
      if SignNeg then
        Value := -V else
        Value := V;
      result := true;
    end;
  end;
  if Next=nil then
    exit;
  while not(U^ in [#0,'&']) do inc(U);
  if U^=#0 then
    Next^ := nil else
    Next^ := U+1; // jump '&'
end;

// - UrlDecodeNeedParameters('price=20.45&where=LastName%3D','PRICE,WHERE') will
// return TRUE
function UrlDecodeNeedParameters(U, CSVUpper: PUTF8Char): boolean;
var tmp: array[0..32] of AnsiChar;
    L: PtrInt;
    Beg: PUTF8Char;
begin
  result := (CSVUpper=nil);
  if result then
    exit; // no parameter to check -> success
  repeat
    L := 0;
    while (CSVUpper^<>#0) and (CSVUpper^<>',') do begin
      tmp[L] := NormToUpper[CSVUpper^];
      if L=high(tmp) then
        exit else // invalid CSV parameter
        inc(L);
      inc(CSVUpper);
    end;
    if L=0 then
      exit; // invalid CSV parameter
    PWord(tmp+L)^ := ord('=');
    Beg := U;
    repeat
      if IdemPChar(U,tmp) then
        break;
      while not(U^ in [#0,'&']) do inc(U);
      if U^=#0 then
        exit else // didn't find tmp in U
        inc(U); // Jump &
    until false;
    U := Beg;
    if CSVUpper^=#0 then
      Break else // no more parameter to check
      inc(CSVUpper); // jump &
  until false;
  result := true; // all parameters found
end;

function UrlDecodeExtended(U, Upper: PUTF8Char; var Value: Extended; Next: PPUTF8Char=nil): boolean;
var tmp: RawUTF8;
    err: integer;
begin
  result := UrlDecodeValue(U, Upper, tmp, Next);
  if result then
    Value := GetExtended(pointer(tmp),err);
end;

function Hash32(const Text: RawByteString): cardinal;
{$ifdef PUREPASCAL} // this code is quite as fast as the optimized asm below
function SubHash(P: PCardinalArray): cardinal;
{$ifdef HASINLINE}inline;{$endif}
var s1,s2: cardinal;
    i, L: PtrInt;
const Mask: array[0..3] of cardinal = (0,$ff,$ffff,$ffffff);
begin
  if P<>nil then begin
    L := PPtrInt(PtrInt(P)-4)^; // fast lenght(Text)
    s1 := 0;
    s2 := 0;
    for i := 1 to L shr 4 do begin // 16 bytes (4 DWORD) by loop - aligned read
      inc(s1,P^[0]);
      inc(s2,s1);
      inc(s1,P^[1]);
      inc(s2,s1);
      inc(s1,P^[2]);
      inc(s2,s1);
      inc(s1,P^[3]);
      inc(s2,s1);
      inc(PtrUInt(P),16);
    end;
    for i := 1 to (L shr 2)and 3 do begin // 4 bytes (DWORD) by loop
      inc(s1,P^[0]);
      inc(s2,s1);
      inc(PtrUInt(P),4);
    end;
    inc(s1,P^[0] and Mask[L and 3]);      // remaining 0..3 bytes
    inc(s2,s1);
    result := s1 xor (s2 shl 16);
  end else
    result := 0;
end;
begin // use a sub function for better code generation under Delphi
  result := SubHash(pointer(Text));
end;
{$else}
asm // our simple and efficient algorithm (ADLER-32 based) is:
    //   while(data) do { s1 := s1+DWORD(data); s2 := s2+s1; }
    //   return (s1 xor (s2 shl 16));
    // this asm code is very optimized for modern pipelined CPU
    or eax,eax
    push ebx
    jz @z
    mov ecx,[eax-4] // ecx = length(Key)
    mov edx,eax     // edx = Text
    xor eax,eax     // eax = s1 = 0
    xor ebx,ebx     // ebx = s2 = 0
    push ecx
    shr ecx,2
    jz @n
    push ecx
    shr ecx,2
    jz @m
    nop; nop
@16:add eax,[edx]   // 16 bytes (4 DWORD) by loop - aligned read
    add ebx,eax
    add eax,[edx+4] // both 'add' are pipelined: every DWORD is processed at once
    add ebx,eax
    add eax,[edx+8]
    add ebx,eax
    add eax,[edx+12]
    add ebx,eax
    dec ecx
    lea edx,edx+16
    jnz @16
@m: pop ecx
    and ecx,3
    jz @n
    nop
@4: add eax,[edx]  // 4 bytes (DWORD) by loop
    add ebx,eax
    dec ecx
    lea edx,edx+4
    jnz @4
@n: pop ecx
    mov edx,[edx] // read last DWORD value
    and ecx,3     // remaining 0..3 bytes
    and edx,dword ptr [@Mask+ecx*4] // trim to DWORD value to 0..3 bytes
    add eax,edx
    add ebx,eax
    shl ebx,16
    xor eax,ebx  // return (s1 xor (s2 shl 16))
@z: pop ebx
    ret
    nop; nop // align @Mask
@Mask: dd 0,$ff,$ffff,$ffffff // to get only relevant byte information
end;
{asm // slower hash implementation (algorithm from IniFiles.TStringHash.HashOf)
    or eax,eax
    jz @z
    push ebx
    mov edx,eax     // edx = Text
    xor eax,eax     // eax = Result
    xor ecx,ecx     // ecx = Result shl 2 = 0
    mov ebx,[edx-4] // ebx = length(Key)
@1: shr eax,$1e     // eax = Result shr (SizeOf(Result) * 8 - 2))
    or ecx,eax      // ecx = ((Result shl 2) or (Result shr (SizeOf(Result) * 8 - 2)))
    movzx eax,byte ptr [edx] // eax = ord(Key[i])
    inc edx
    xor eax,ecx     // eax = () xor ord(Key[i])
    dec ebx
    lea ecx,[eax*4] // ecx = Result shl 2
    jnz @1
    pop ebx
@z:
end;}
{$endif}

function Hash32(Data: pointer; Len: integer): cardinal;
{$ifdef PUREPASCAL} // this code is quite as fast as the optimized asm below
function SubHash(P: PCardinalArray; L: integer): cardinal;
{$ifdef HASINLINE}inline;{$endif}
var s1,s2: cardinal;
    i: PtrInt;
const Mask: array[0..3] of cardinal = (0,$ff,$ffff,$ffffff);
begin
  if P<>nil then begin
    s1 := 0;
    s2 := 0;
    for i := 1 to L shr 4 do begin // 16 bytes (4 DWORD) by loop - aligned read
      inc(s1,P^[0]);
      inc(s2,s1);
      inc(s1,P^[1]);
      inc(s2,s1);
      inc(s1,P^[2]);
      inc(s2,s1);
      inc(s1,P^[3]);
      inc(s2,s1);
      inc(PtrUInt(P),16);
    end;
    for i := 1 to (L shr 2)and 3 do begin // 4 bytes (DWORD) by loop
      inc(s1,P^[0]);
      inc(s2,s1);
      inc(PtrUInt(P),4);
    end;
    inc(s1,P^[0] and Mask[L and 3]);      // remaining 0..3 bytes
    inc(s2,s1);
    result := s1 xor (s2 shl 16);
  end else
    result := 0;
end;
begin // use a sub function for better code generation under Delphi
  result := SubHash(Data,Len);
end;
{$else}
asm // our simple and efficient algorithm (ADLER-32 based) is:
    //   while(data) do { s1 := s1+DWORD(data); s2 := s2+s1; }
    //   return (s1 xor (s2 shl 16));
    // this asm code is very optimized for modern pipelined CPU
    or eax,eax
    push ebx
    jz @z
    mov ecx,edx     // ecx = length(Data)
    mov edx,eax     // edx = Data
    xor eax,eax     // eax = s1 = 0
    xor ebx,ebx     // ebx = s2 = 0
    push ecx
    shr ecx,2
    jz @n
    push ecx
    shr ecx,2
    jz @m
    nop; nop
@16:add eax,[edx]   // 16 bytes (4 DWORD) by loop - aligned read
    add ebx,eax
    add eax,[edx+4] // both 'add' are pipelined: every DWORD is processed at once
    add ebx,eax
    add eax,[edx+8]
    add ebx,eax
    add eax,[edx+12]
    add ebx,eax
    dec ecx
    lea edx,edx+16
    jnz @16
@m: pop ecx
    and ecx,3
    jz @n
    nop
@4: add eax,[edx]  // 4 bytes (DWORD) by loop
    add ebx,eax
    dec ecx
    lea edx,edx+4
    jnz @4
@n: pop ecx
    mov edx,[edx] // read last DWORD value
    and ecx,3     // remaining 0..3 bytes
    and edx,dword ptr [@Mask+ecx*4] // trim to DWORD value to 0..3 bytes
    add eax,edx
    add ebx,eax
    shl ebx,16
    xor eax,ebx  // return (s1 xor (s2 shl 16))
@z: pop ebx
    ret
    nop; nop // align @Mask
@Mask: dd 0,$ff,$ffff,$ffffff // to get only relevant byte information
end;
{$endif}

function GetBit(const Bits; aIndex: PtrInt): boolean;
{$ifdef PUREPASCAL} 
begin
{$ifdef CPU64}
  result := PInt64Array(@Bits)^[aIndex shr 6] and (Int64(1) shl (aIndex and 63)) <> 0;
{$else}
  result := PIntegerArray(@Bits)^[aIndex shr 5] and (1 shl (aIndex and 31)) <> 0;
{$endif}
end;
{$else}
asm
  bt [eax],edx // use very fast i386 bit statement
  sbb eax,eax
  and eax,1
end;
{$endif}

procedure SetBit(var Bits; aIndex: PtrInt);
{$ifdef PUREPASCAL} 
begin
{$ifdef CPU64}
  PInt64Array(@Bits)^[aIndex shr 6] := PInt64Array(@Bits)^[aIndex shr 6]
    or (Int64(1) shl (aIndex and 63));
{$else}
  PIntegerArray(@Bits)^[aIndex shr 5] := PIntegerArray(@Bits)^[aIndex shr 5]
    or (1 shl (aIndex and 31));
{$endif}
end;
{$else}
asm
  bts [eax],edx // use very fast i386 bit statement
end;
{$endif}

procedure UnSetBit(var Bits; aIndex: PtrInt);
{$ifdef PUREPASCAL} 
begin
  PIntegerArray(@Bits)^[aIndex shr 5] := PIntegerArray(@Bits)^[aIndex shr 5]
    and not (1 shl (aIndex and 31));
end;
{$else}
asm
  btr [eax],edx // use very fast i386 bit statement
end;
{$endif}

function GetBit64(const Bits; aIndex: PtrInt): boolean;
{$ifdef PUREPASCAL}
begin
  if PtrUInt(aIndex)>63 then
    result := false else
{$ifdef CPU64}
    result := PInt64(@Bits)^ and (Int64(1) shl (aIndex and 63)) <> 0;
{$else}
    result := PIntegerArray(@Bits)^[aIndex shr 5] and (1 shl (aIndex and 31)) <> 0;
{$endif}
end;
{$else}
asm
    cmp edx,64
    jae @z
    bt [eax],edx  // use very fast i386 bit statement
    sbb eax,eax
    and eax,1
    ret
@z: xor eax,eax
end;
{$endif}

procedure SetBit64(var Bits: Int64; aIndex: PtrInt);
{$ifdef PUREPASCAL} 
begin
  if PtrUInt(aIndex)<=63 then
{$ifdef CPU64}
    PInt64Array(@Bits)^[aIndex shr 6] := PInt64Array(@Bits)^[aIndex shr 6]
      or (Int64(1) shl (aIndex and 63));
{$else}
    PIntegerArray(@Bits)^[aIndex shr 5] := PIntegerArray(@Bits)^[aIndex shr 5]
      or (1 shl (aIndex and 31));
{$endif}
end;
{$else}
asm
  cmp edx,64
  jae @z
  bts [eax],edx  // use very fast i386 bit statement
@z:
end;
{$endif}

procedure UnSetBit64(var Bits: Int64; aIndex: PtrInt);
{$ifdef PUREPASCAL} 
begin
  if PtrUInt(aIndex)<=63 then
{$ifdef CPU64}
    PInt64Array(@Bits)^[aIndex shr 6] := PInt64Array(@Bits)^[aIndex shr 6]
      and not(Int64(1) shl (aIndex and 63));
{$else}
    PIntegerArray(@Bits)^[aIndex shr 5] := PIntegerArray(@Bits)^[aIndex shr 5]
      and not (1 shl (aIndex and 31));
{$endif}
end;
{$else}
asm
  cmp edx,64
  jae @z
  btr [eax],edx // use very fast i386 bit statement
@z:
end;
{$endif}

function GetBitsCount(const Bits; Count: PtrInt): integer;
{$ifdef PUREPASCAL}
begin
  result := 0;
  while Count>0 do begin
    dec(Count);
    if GetBit(Bits,Count) then
      inc(result);
  end;
end;
{$else}
asm
    xor ecx,ecx
@1: or edx,edx
    jz @n
    dec edx
    bt [eax],edx
    adc ecx,0
    jmp @1
@n: mov eax,ecx
end;
{$endif}

function Curr64ToStr(Value: Int64): RawUTF8;
var P,L: PtrInt;
    Dec: cardinal;
begin
  // 1. get numerical value (*1000)
  result := {$ifndef ENHANCEDRTL}Int64ToUtf8{$else}IntToStr{$endif}(Value); // very fast
  // 2. add '.' character at proper position
  P := PtrInt(result);
  L := PInteger(P-4)^+1;
  case L of // only decimals: returns '0.1234'
    2: begin result := '0.000'+result; exit; end;
    3: begin result := '0.00'+result;  exit; end;
    4: begin result := '0.0'+result;   exit; end;
    5: begin result := '0.'+result;    exit; end;
  end;
  Dec := PCardinal(P+L-5)^; // 4 last digits = 4 decimals
  if Dec=ord('0')+ord('0')shl 8+ord('0')shl 16+ord('0')shl 24 then // no decimal
    SetLength(result,L-5) else
  if Dec and $ffff0000=ord('0')shl 16+ord('0')shl 24 then begin // 2 decimals
    SetLength(result,L-2);
    pCardinal(PtrInt(result)+L-5)^ := ord('.')+word(Dec)shl 8;
  end else begin // 4 decimals
    SetLength(result,L);
    P := PtrInt(result)+L-5;
    PAnsiChar(P)^ := '.'; // '12345678' -> '1234.5678'
    pCardinal(P+1)^ := Dec;
  end;
end;

function Curr64ToPChar(Value: Int64; Dest: PUTF8Char): PtrInt;
var tmp: RawUTF8;
    P,L: PtrInt;
    Dec: cardinal;
begin
  // 1. get numerical value (*1000)
  tmp := {$ifndef ENHANCEDRTL}Int64ToUtf8{$else}IntToStr{$endif}(Value); // very fast
  // 2. add '.' character at proper position
  P := PtrInt(tmp);
  L := PInteger(P-4)^-4;
  if L<=0 then begin // only decimals: returns '0.1234'
    PCardinal(Dest)^ := ord('0')+ord('.')shl 8+ord('0')shl 16+ord('0')shl 24;
    Dest[4] := '0';
    PCardinal(Dest-L+2)^ := PCardinal(P)^; // set decimals
    result := 6;
    exit;
  end;
  move(pointer(P)^,Dest^,L);
  inc(Dest,L);
  Dec := PCardinal(P+L)^; // 4 last digits = 4 decimals
  if Dec=ord('0')+ord('0')shl 8+ord('0')shl 16+ord('0')shl 24 then
    result := L else // no decimal
  if Dec and $ffff0000=ord('0')shl 16+ord('0')shl 24 then begin // 2 decimals
    pCardinal(Dest)^ := ord('.')+word(Dec)shl 8;
    result := L+3;
  end else begin // 4 decimals
    Dest^ := '.'; // '12345678' -> '1234.5678'
    pCardinal(Dest+1)^ := Dec;
    result := L+5;
  end;
end;

function StrToCurr64(P: PUTF8Char): Int64;
var c: cardinal;
    minus: boolean;
    Dec: cardinal;
begin
  result := 0;
  if P=nil then
    exit;
  if P^=' ' then repeat inc(P) until P^<>' ';
  if P^='-' then begin
    minus := true;
    repeat inc(P) until P^<>' ';
  end else begin
    minus := false;
    if P^='+' then
      repeat inc(P) until P^<>' ';
  end;
  c := byte(P^)-48;
  if c>9 then
    exit;
  Int64Rec(result).Lo := c;
  Dec := 0;
  inc(P);
  repeat
    if P^<>'.' then begin
      c := byte(P^)-48;
      if c>9 then
        break;
      result := result shl 3+result+result; // fast result := result*10
      inc(result,c);
      inc(P);
      if Dec<>0 then begin
        inc(Dec);
        if Dec<5 then continue else break;
      end;
    end else begin
      inc(Dec);
      inc(P);
    end;
  until false;
  if Dec<>5 then // Dec=5 most of the time
  case Dec of
  0,1: result := result*10000;
  2:   result := result*1000; // result shl 10-result shl 4-result shl 3;
  3:   result := result shl 6+result shl 5+result shl 2; // result := result*100
  4:   result := result shl 3+result+result; // fast result := result*10
  end;
  if minus then
    result := -result;
end;

function TrimLeftLowerCase(V: PShortString): RawUTF8;
{$ifdef NODELPHIASM}
var P: PAnsiChar;
    L: integer;
begin
  L := length(V^);
  P := @V^[1];
  while (L>0) and (P^ in ['a'..'z']) do begin
    inc(P);
    dec(L);
  end;
  if L=0 then
    result := V^ else
    SetString(result,P,L);
end;
{$else}
asm // eax=V
    xor cl,cl
    push edx // save result RawUTF8
    or eax,eax
    jz @2 // avoid GPF
    lea edx,eax+1
    mov cl,[eax]
@1: mov ch,[edx] // edx=source cl=length
    sub ch,'a'
    sub ch,'z'-'a'
    ja @2 // not a lower char -> create a result string starting at edx
    inc edx
    dec cl
    jnz @1
    mov cl,[eax]
    lea edx,eax+1  // no UpperCase -> retrieve full text (result := V^)
@2: pop eax
    movzx ecx,cl
{$ifdef UNICODE}
    push CP_UTF8 // UTF-8 code page for Delphi 2009/2010 + call below, not jump
    call System.@LStrFromPCharLen // eax=Dest edx=Source ecx=Length
    ret // we need a call just above for right push CP_UTF8 retrieval
{$else}
    jmp System.@LStrFromPCharLen // eax=dest edx=source ecx=length(source)
{$endif}
end;
{$endif}

function UnCamelCase(const S: RawUTF8): RawUTF8; overload;
begin
  result := '';
  if S='' then
    exit;
  SetLength(result,PInteger(PtrInt(S)-4)^*2); // max length
{$ifdef UNICODE2} // not needed: SetLength() did already set the codepage
  PWord(PtrInt(result)-12)^ := CP_UTF8; // use only SetLength() -> force set code page
{$endif}
  SetLength(result,UnCamelCase(pointer(result),pointer(S)));
end;

function UnCamelCase(D, P: PUTF8Char): integer; overload;
var Space, SpaceBeg, DBeg: PUTF8Char;
label Next;
begin
  Space := D;
  DBeg := D;
  SpaceBeg := D;
  if (D<>nil) and (P<>nil) then // avoid GPF
  repeat
    repeat
      D^ := P^;
      inc(P);
      inc(D);
    until not (P^ in ['A'..'Z']);
    if P^=#0 then break; // no lowercase conversion of last fully uppercased word
    while P^ in ['a'..'z','0'..'9'] do begin
      D^  := P^;
      inc(D);
      inc(P);
    end;
    if P^='_' then
    if P[1]='_' then begin
      D^ := ':';
      inc(P);
      inc(D);
      goto Next;
    end else begin
      PWord(D)^ := ord(' ')+ord('-')shl 8;
      inc(D,2);
Next: if Space=SpaceBeg then
        SpaceBeg := D+1;
      inc(P);
      Space := D+1;
    end else
      Space := D;
    if P^=#0 then break;
    D^ := ' ';
    inc(D);
  until false;
  while Space>SpaceBeg do begin
    if (Space^ in ['A'..'Z']) then
      inc(Space^,32); // lowercase conversion of not last fully uppercased word
    dec(Space);
  end;
  result := D-DBeg;
end;

function CharSetToCodePage(CharSet: integer): cardinal;
begin
  case CharSet of
    SHIFTJIS_CHARSET:   result := 932;
    HANGEUL_CHARSET:    result := 949;
    GB2312_CHARSET:     result := 936;
    HEBREW_CHARSET:     result := 1255;
    ARABIC_CHARSET:     result := 1256;
    GREEK_CHARSET:      result := 1253;
    TURKISH_CHARSET:    result := 1254;
    VIETNAMESE_CHARSET: result := 1258;
    THAI_CHARSET:       result := 874;
    EASTEUROPE_CHARSET: result := 1250;
    RUSSIAN_CHARSET:    result := 1251;
    BALTIC_CHARSET:     result := 1257;
  else result := CODEPAGE_US; // default is ANSI_CHARSET = iso-8859-1 = windows-1252
  end;
end;

function CodePageToCharSet(CodePage: Cardinal): Integer;
begin
  case CodePage of
    932:  result := SHIFTJIS_CHARSET;
    949:  result := HANGEUL_CHARSET;
    936:  result := GB2312_CHARSET;
    1255: result := HEBREW_CHARSET;
    1256: result := ARABIC_CHARSET;
    1253: result := GREEK_CHARSET;
    1254: result := TURKISH_CHARSET;
    1258: result := VIETNAMESE_CHARSET;
    874:  result := THAI_CHARSET;
    1250: result := EASTEUROPE_CHARSET;
    1251: result := RUSSIAN_CHARSET;
    1257: result := BALTIC_CHARSET;
  else result := ANSI_CHARSET; // default is iso-8859-1 = windows-1252
  end;
end;

function GetMimeContentType(Content: Pointer; Len: integer): RawUTF8;
begin // see http://www.garykessler.net/library/file_sigs.html for magic numbers
  result := '';
  if (Content<>nil) and (Len>4) then
    case PCardinal(Content)^ of
    $04034B50: Result := 'application/zip'; // 50 4B 03 04
    $46445025: Result := 'application/pdf'; //  25 50 44 46 2D 31 2E
    $21726152: Result := 'application/x-rar-compressed'; // 52 61 72 21 1A 07 00
    $AFBC7A37: Result := 'application/x-7z-compressed'; // 37 7A BC AF 27 1C
    $75B22630: Result := 'audio/x-ms-wma'; // 30 26 B2 75 8E 66
    $9AC6CDD7: Result := 'video/x-ms-wmv'; // D7 CD C6 9A 00 00
    $474E5089: Result := 'image/png'; // 89 50 4E 47 0D 0A 1A 0A
    $38464947: Result := 'image/gif'; // 47 49 46 38
    $002A4949, $2A004D4D, $2B004D4D:
      Result := 'image/tiff'; // 49 49 2A 00 or 4D 4D 00 2A or 4D 4D 00 2B
    $E011CFD0: // Microsoft Office applications D0 CF 11 E0 = DOCFILE
      if Len>600 then
      case PWordArray(Content)^[256] of // at offset 512
        $A5EC: Result := 'application/msword'; // EC A5 C1 00
        $FFFD: // FD FF FF
          case PByteArray(Content)^[516] of
            $0E,$1C,$43: Result := 'application/vnd.ms-powerpoint';
            $10,$1F,$20,$22,$23,$28,$29: Result := 'application/vnd.ms-excel';
          end;
      end;
    else
      case PCardinal(Content)^ and $00ffffff of
        $685A42: Result := 'application/bzip2'; // 42 5A 68
        $088B1F: Result := 'application/gzip'; // 1F 8B 08
        $492049: Result := 'image/tiff'; // 49 20 49
        $FFD8FF: Result := 'image/jpeg'; // FF D8 FF DB/E0/E1/E2/E3/E8
        else
          case PWord(Content)^ of
            $4D42: Result := 'image/bmp'; // 42 4D
          end;
      end;
    end;
  if Result='' then
    Result := 'application/octet-stream';
end;

{$ifdef MSWINDOWS}

{ TFileVersion }

procedure TFileVersion.RetrieveApplicationVersion(const FileName: TFileName;
  DefaultVersion: integer);
var Size, Size2: DWord;
    Pt: Pointer;
    Info: ^TVSFixedFileInfo;
    FileTime: TFILETIME;
    SystemTime: TSYSTEMTIME;
begin
  Major := DefaultVersion;
  Minor := 0;
  Release := 0;
  Build := 0;
  BuildDateTime := 0;
  if FileName='' then
    exit;
  // GetFileVersionInfo modifies the filename parameter data while parsing.
  // Copy the string const into a local variable to create a writeable copy.
  Size := GetFileVersionInfoSize(pointer(@FileName[1]), Size2);
  if Size>0 then begin
    GetMem(Pt, Size);
    try
      GetFileVersionInfo(pointer(FileName), 0, Size, Pt);
      VerQueryValue(Pt, '\', pointer(Info), Size2);
      with Info^ do begin
        Major := dwFileVersionMS shr 16;
        Minor := word(dwFileVersionMS);
        Release := dwFileVersionLS shr 16;
        Build := word(dwFileVersionLS);
        BuildYear := 2010;
        if (dwFileDateLS<>0) and (dwFileDateMS<>0) then begin
          FileTime.dwLowDateTime:= dwFileDateLS; // built date from version info
          FileTime.dwHighDateTime:= dwFileDateMS;
          FileTimeToSystemTime(FileTime, SystemTime);
          BuildDateTime := EncodeDate(
            SystemTime.wYear,SystemTime.wMonth,SystemTime.wDay);
          BuildYear := SystemTime.wYear;
        end;
      end;
    finally
      Freemem(Pt);
    end;
  end;
  Main := IntToStr(Major) + '.' + IntToStr(Minor);
  Detailed := Main+ '.' + IntToStr(Release) + '.' + IntToStr(Build);
  if BuildDateTime=0 then // get build date from file age
    BuildDateTime := FileAgeToDateTime(FileName);
end;

function TFileVersion.Version32: integer;
begin
  result := Major shl 16+Minor shl 8+Release;
end;

{$else}

function GetTickCount: Cardinal;
begin
  result := (Int64(clock)*1000) div CLOCKS_PER_SEC; // ms result
end;

{$endif MSWINDOWS}

{$ifndef LVCL}

{ THeapMemoryStream = faster TMemoryStream using FastMM4 heap, not windows.GlobalAlloc() }

const
  MemoryDelta = $8000; // 32 KB granularity (must be a power of 2)

function THeapMemoryStream.Realloc(var NewCapacity: Integer): Pointer;
// allocates memory from Delphi heap (FastMM4) and not windows.Global*()
// and uses bigger growing size -> a lot faster
var i: PtrInt;
begin
  if NewCapacity>0 then begin
    i := Seek(0,soFromCurrent); // no direct access to fSize -> use Seek() trick
    if NewCapacity=Seek(0,soFromEnd) then begin // avoid ReallocMem() if just truncate
      result := Memory;
      Seek(i,soFromBeginning);
      exit;
    end;
    NewCapacity := (NewCapacity + (MemoryDelta - 1)) and not (MemoryDelta - 1);
    Seek(i,soFromBeginning);
  end;
  Result := Memory;
  if NewCapacity <> Capacity then begin
    if NewCapacity = 0 then begin
      FreeMem(Memory);
      Result := nil;
    end else begin
      if Capacity = 0 then
        GetMem(Result, NewCapacity) else
        if NewCapacity > Capacity then // only realloc if necessary (grow up)
          ReallocMem(Result, NewCapacity) else
          NewCapacity := Capacity; // same capacity as before
      if Result = nil then
        raise EStreamError.Create('THeapMemoryStream'); // memory allocation bug
    end;
  end;
end;

{$endif LVCL}


{ TSortedWordArray }

function FastLocateWordSorted(P: PWordArray; R: integer; Value: word): PtrInt;
var L,cmp: PtrInt;
begin
  if R<0 then
    result := 0 else begin
    L := 0;
    repeat
      result := (L + R) shr 1;
      cmp := P^[result]-Value;
      if cmp=0 then begin
        result := -result-1; // return -(foundindex+1) if already exists
        exit;
      end;
      if cmp<0 then
        L := result + 1 else
        R := result - 1;
    until (L > R);
    while (result>=0) and (P^[result]>=Value) do dec(result);
    result := result+1; // return the index where to insert
  end;
end;

function TSortedWordArray.Add(aValue: Word): PtrInt;
begin
  result := FastLocateWordSorted(pointer(Values),Count-1,aValue);
  if result<0 then // aValue already exists in Values[] -> fails
    exit;
  if Count=length(Values) then
    SetLength(Values,Count+100);
  if result<Count then
    move(Values[result],Values[result+1],(Count-result)*2) else
    result := Count;
  Values[result] := aValue;
  inc(Count);
end;

function TSortedWordArray.IndexOf(aValue: Word): PtrInt;
var L,R: PtrInt;
    cmp: integer;
begin
  L := 0;
  R := Count-1;
  if 0<=R then
  repeat
    result := (L + R) shr 1;
    cmp := Values[result]-aValue;
    if cmp=0 then
      exit else
    if cmp<0 then
      L := result + 1 else
      R := result - 1;
  until (L > R);
  result := -1;
end;


{ ****************** text buffer and JSON functions and classes ********* }

{ TTextWriter }

procedure TTextWriter.Add(Value: integer);
var tmp: array[0..15] of AnsiChar;
    P: PAnsiChar;
begin
  P := StrInt32(@tmp[15],Value);
  AddNoJSONEscape(P,@tmp[15]-P);
end;

procedure TTextWriter.Add(Value: double; decimals: integer);
var S: ShortString;
begin
  AddNoJSONEscape(@S[1],ExtendedToString(S,Value,decimals));
end;

procedure TTextWriter.Add(Value: Int64);
var tmp: array[0..23] of AnsiChar;
    P: PAnsiChar;
begin
{$ifdef CPU64}
  P := StrInt32(@tmp[23],Value); // StrInt32 use PtrInt, i.e. Int64
{$else}
  P := StrInt64(@tmp[23],Value);
{$endif}
  AddNoJSONEscape(P,@tmp[23]-P);
end;

procedure TTextWriter.Add(c: AnsiChar);
begin
  if B=BEnd then
    FlushInc else
    inc(B);
  B^ := c;
end;

procedure TTextWriter.Add2(Value: integer);
begin
  if B+2>=BEnd then
    FlushInc else
    inc(B);
  if cardinal(Value)>99 then
    pCardinal(B)^ := $3030+ord(',')shl 16 else     // '00,' if overflow
    pCardinal(B)^ := TwoDigitLookupW[Value]+ord(',')shl 16;
  inc(B,2);
end;

procedure TTextWriter.Add4(Value: integer);
begin
  if B+4>=BEnd then
    FlushInc else
    inc(B);
  if cardinal(Value)>9999 then
    pCardinal(B)^ := $30303030 else // '0000,' if overflow
    YearToPChar(Value,B);
  inc(B,4);
  B^ := ',';
end;

procedure TTextWriter.Add3(Value: integer);
begin
  if B+3>=BEnd then
    FlushInc else
    inc(B);
  if cardinal(Value)>999 then
    pCardinal(B)^ := $303030 else // '0000,' if overflow
    pCardinal(B)^ := TwoDigitLookupW[Value div 10]+
      ord(Value mod 10+48)shl 16;
  inc(B,3);
  B^ := ',';
end;

procedure TTextWriter.AddCR;
begin
  if B+1>=BEnd then
    FlushInc else
    inc(B);
  pWord(B)^ := 13+10 shl 8; // CR + LF
  inc(B);
end;


procedure TTextWriter.AddCSVIntegers(const Integers: array of Integer);
var i: integer;
begin
  if length(Integers)=0 then
    exit;
  for i := 0 to high(Integers) do begin
    Add(Integers[i]);
    Add(',');
  end;
  CancelLastComma;
end;

procedure TTextWriter.Add(Format: PWinAnsiChar; const Values: array of const);
var ValuesIndex: integer;
begin // we put const char > #127 as #??? -> asiatic MBCS codepage OK
  if Format=nil then
    exit;
  ValuesIndex := 0;
  repeat
    repeat
      case Format^ of
      #0: exit;
      #13, #164: AddCR; // CR, -> add CR,LF
      #167: if B^=',' then dec(B); // 
      '$','%',#163,#181: break; // $,%,,
      else begin
        if B=BEnd then
          FlushInc else
          inc(B);
        B^ := Format^;
      end;
      end;
      inc(Format);
    until false;
    // add next value as text
    if ValuesIndex<=high(Values) then // missing value will display nothing
    case Format^ of
    '%': with Values[ValuesIndex] do
         case Vtype of
           vtBoolean:    Add(integer(VBoolean));
           vtString:     AddNoJSONEscape(@VString^[1],ord(VString^[0])); // expect RawUTF8
           vtAnsiString: if VAnsiString<>nil then // expect RawUTF8
             AddNoJSONEscape(VAnsiString,length(AnsiString(VAnsiString)));
{$ifdef UNICODE}
           vtUnicodeString: if VUnicodeString<>nil then // convert to UTF-8
             AddNoJSONEscapeW(VUnicodeString,length(UnicodeString(VUnicodeString)));
{$endif}   vtPChar:      AddNoJSONEscape(VPChar); // will call StrLen()
           vtChar,
           vtWideChar:   AddNoJSONEscape(@VChar,1); // only ansi value
           vtInteger:    Add(VInteger);
           vtInt64:      Add(VInt64^);
           vtExtended:   Add(VExtended^);
         end;
    '$': with Values[ValuesIndex] do
           if Vtype=vtInteger then Add2(VInteger);
    #163: with Values[ValuesIndex] do // 
           if Vtype=vtInteger then Add4(VInteger);
    #181: with Values[ValuesIndex] do // 
           if Vtype=vtInteger then Add3(VInteger);
    end;
    inc(Format);
    inc(ValuesIndex);
  until false;
end;

procedure TTextWriter.AddLine(const Text: shortstring);
begin
  if B+ord(Text[0])+2>=BEnd then
    FlushInc else
    inc(B);
  move(Text[1],B^,ord(Text[0]));
  inc(B,ord(Text[0]));
  pWord(B)^ := 13+10 shl 8; // CR + LF
  inc(B);
end;

procedure TTextWriter.AddNoJSONEscape(P: Pointer; Len: integer);
var i: integer;
label Escape;
begin
  if P=nil then exit;
  if Len=0 then
    Len := StrLen(PUTF8Char(P));
  if Len>0 then begin // no JSONify:
    // inc(B) is done first to allow CancelLastChar
    if B=BEnd then
      FlushInc else
      inc(B);
    if Len=1 then begin
      // add one char
      B^ := PAnsiChar(P)^;
      inc(PtrInt(P));
    end else begin
      // add two or more chars
      repeat
        // guess biggest size to be added into buf^ at once
        i := BEnd-B;
        if Len<i then
          i := Len;
        // add UTF-8 bytes
        move(P^,B^,i);
        inc(PtrInt(P),i);
        inc(B,i);
        dec(Len,i);
        if Len=0 then break;
        fStream.Write(buf,B-buf); // FlushInc writes B-buf+1 -> special one here
        B := buf;
      until false;
      dec(B); // allow CancelLastChar
    end;
  end;
end;

procedure TTextWriter.AddNoJSONEscapeW(P: Pointer; WideCharCount: integer);
var tmp: RawUTF8;
begin
  tmp := RawUnicodeToUtf8(P,WideCharCount); // fast Unicode -> UTF-8 conversion
  AddNoJSONEscape(pointer(tmp),length(tmp));
end;


procedure TTextWriter.AddJSONEscape(P: Pointer; Len: PtrInt);
var i: PtrInt;
    c: ansichar;
    v: PtrUInt;
label Escape;
begin
  if P=nil then exit;
  if Len=0 then
    Len := MaxInt;
  for i := 1 to Len do begin
    // inc(B) is done first to allow CancelLastChar
    if B=BEnd then
      FlushInc else
      inc(B);
    // escape chars, according to http://www.ietf.org/rfc/rfc4627.txt
    case PAnsiChar(P)^ of
      #0: begin
        dec(B); break; end;
      #8: begin
        c := 'b'; goto Escape; end;
      #9: begin
        c := 't'; goto Escape; end;
      #$a: begin
        c := 'n'; goto Escape; end;
      #$c: begin
        c := 'f'; goto Escape; end;
      #$d: begin
        c := 'r'; goto Escape; end;
      '\','/','"': begin // litterals
        c := PAnsiChar(P)^;
Escape: B^ := '\';
        if B=BEnd then  // inlined: avoid endless loop
          FlushInc else
          inc(B);
        B^ := c;
      end;
      #01..#07, #$0b, #$0e..#$1f: begin // characters below ' ', #7 e.g.
        B^ := '\';
        v := byte(PAnsiChar(P)^);
        BufUnicode[3] := HexChars[v shr 4];
        BufUnicode[4] := HexChars[v and $F];
        AddNoJSONEscape(@BufUnicode,5); // 'u0007' e.g.
      end;
      else
        B^ := PAnsiChar(P)^;
    end;
    inc(PtrInt(P));
  end
end;

procedure TTextWriter.AddJSONEscapeUnicode(P: PWord; Len: PtrInt);
var i: PtrInt;
    c: ansichar;
    v: PtrUInt;
label Escape;
begin
  if P=nil then exit;
  if Len=0 then
    Len := MaxInt;
  for i := 1 to Len do begin
    // inc(B) is done first to allow CancelLastChar
    if B=BEnd then
      FlushInc else
      inc(B);
    // escape chars, according to http://www.ietf.org/rfc/rfc4627.txt
    case P^ of
      0: begin
        dec(B); break; end;
      8: begin
        c := 'b'; goto Escape; end;
      9: begin
        c := 't'; goto Escape; end;
      10: begin
        c := 'n'; goto Escape; end;
      12: begin
        c := 'f'; goto Escape; end;
      13: begin
        c := 'r'; goto Escape; end;
      ord('\'),ord('/'),ord('"'): begin // litterals 34,47,92
        c := AnsiChar(ord(P^));
Escape: B^ := '\';
        if B=BEnd then  // inlined: avoid endless loop
          FlushInc else
          inc(B);
        B^ := c;
      end;
      32,33,35..46,48..91,93..126:
        B^ := AnsiChar(ord(P^)); // direct store 7 bits ASCII
      1..7, 11, 14..31: begin // characters below ' ', #7 e.g.
        B^ := '\';
        v := byte(PAnsiChar(P)^);
        BufUnicode[3] := HexChars[v shr 4];
        BufUnicode[4] := HexChars[v and $F];
        AddNoJSONEscape(@BufUnicode,5); // 'u0007' e.g.
      end;
      else // characters higher than #126 -> UTF-8 encode
        AddNoJSONEscape(BufUnicode+5,
          UnicodeCharToUtf8(@BufUnicode[5],ord(P^)));
    end;
    inc(P);
  end
end;

procedure TTextWriter.AddJSONEscape(const V: TVarRec);
begin
  with V do
  case VType of
    vtString, vtAnsiString,{$ifdef UNICODE}vtUnicodeString,{$endif}
    vtPChar, vtChar, vtWideChar: begin
      Add('"');
      case VType of
        vtString:     AddJSONEscape(@VString^[1],ord(VString^[0]));
        vtAnsiString: AddJSONEscape(pointer(RawUTF8(VAnsiString)));
    {$ifdef UNICODE}
        vtUnicodeString: AddJSONEscapeUnicode(pointer(string(VUnicodeString)));
    {$endif}
        vtPChar:      AddJSONEscape(VPChar);
        vtChar:       AddJSONEscape(@VChar,1);
        vtWideChar:   AddJSONEscapeUnicode(@VWideChar,1);
      end;
      Add('"');
    end;
    vtInteger:  Add(VInteger);
    vtInt64:    Add(VInt64^);
    vtExtended: Add(VExtended^,15);
  end;
end;

procedure TTextWriter.AddString(const Text: RawUTF8);
begin
  if PtrInt(Text)<>0 then
    AddNoJSONEscape(pointer(Text),PInteger(PtrInt(Text)-4)^);
end;

procedure TTextWriter.CancelAll;
begin
  fStream.Position := fInitialStreamPosition;
  B := pointer(buf-1);
end;

procedure TTextWriter.CancelLastChar;
begin
  dec(B);
end;

procedure TTextWriter.CancelLastComma;
begin
  if B^=',' then
    dec(B);
end;

constructor TTextWriter.Create(aStream: TStream);
begin
  if aStream=nil then
    CreateOwnedStream else begin
    fStream := aStream;
    fInitialStreamPosition := fStream.Seek(0,soFromCurrent);
    B := pointer(buf-1);
    BEnd := pointer(buf+high(buf));
    BufUnicode := 'u0000';
  end;
end;

constructor TTextWriter.CreateOwnedStream;
begin
  Create(THeapMemoryStream.Create);
  fStreamIsOwned := true;
end;

destructor TTextWriter.Destroy;
begin
  if fStreamIsOwned then
    FreeAndNil(fStream);
  inherited;
end;

procedure TTextWriter.Flush;
begin
  fStream.Write(buf,B-buf+1);
  B := pointer(buf-1);
end;

procedure TTextWriter.FlushInc;
begin
  fStream.Write(buf,B-buf+1);
  B := buf;
end;

function TTextWriter.GetLength: integer;
begin
  if self=nil then
    result := 0 else
    result := (B-buf+1)-fInitialStreamPosition+fStream.Seek(0,soFromCurrent);
end;

function TTextWriter.Text: RawUTF8;
begin
  if not fStream.InheritsFrom(TMemoryStream) then
    result := '' else begin
    Flush;
    SetString(result,PAnsiChar(TMemoryStream(fStream).Memory)+fInitialStreamPosition,
      TMemoryStream(fStream).Seek(0,soFromCurrent)-fInitialStreamPosition);
  end;
end;

procedure TTextWriter.WrHex(P: PAnsiChar; Len: integer);
var n: integer;
begin
  Flush;
  repeat
    if Len>length(buf)div 2 then
      n := length(buf)div 2 else
      n := Len;
    SynCommons.BinToHex(P,buf,n);
    fStream.Write(buf,n*2);
    inc(P,n);
    dec(Len,n);
  until Len=0;
end;


function JSONEncode(const NameValuePairs: array of const;
  TempMemoryStream: TMemoryStream=nil): RawUTF8;
var A, n: PtrInt;
    W: TTextWriter;
begin
  result := '{}'; // return void JSON object on error
  n := high(NameValuePairs) shr 1;
  if n=0 then
    exit;
  if TempMemoryStream<>nil then
    TempMemoryStream.Seek(0,soFromBeginning);
  W := TTextWriter.Create(TempMemoryStream);
  try
    W.Add('{');
    for A := 0 to n do begin
      W.AddJSONEscape(NameValuePairs[A*2]);
      W.Add(':');
      W.AddJSONEscape(NameValuePairs[A*2+1]);
      W.Add(',');
    end;
    W.CancelLastComma;
    W.Add('}');
    result := W.Text;
  finally
    W.Free
  end;
end;

procedure JSONDecode(var JSON: RawUTF8;
  const UpperNames: array of PUTF8Char; var Values: TPUtf8CharDynArray);
var n, i: PtrInt;
    P, Name, Value: PUTF8Char;
    EndOfObject: AnsiChar;
begin
  n := length(UpperNames);
  SetLength(Values,n);
  fillchar(Values[0],n*sizeof(PUTF8Char),0); // SetLength() could
  dec(n);
  P := pointer(JSON);
  if P=nil then
    exit;
  while P^<>'{' do
    if P^=#0 then
      exit else
      inc(P);
  inc(P); // jump {
  repeat
    Name := GetJSONField(P,P,nil,@EndOfObject);
    if (P=nil) or (Name=nil) or (EndOfObject<>':') then
      exit; // invalid Name:Value separator
    Value := GetJSONField(P,P,nil,@EndOfObject);
    if (Value=nil) or not(EndOfObject in [',','}']) then
      exit; // invalid item separator
    for i := 0 to n do
      if IdemPChar(Name,UpperNames[i]) then begin
        Values[i] := Value;
        break;
      end;
  until (P=nil) or (EndOfObject='}');
end;

function JSONDecode(var JSON: RawUTF8; const aUpperName: RawUTF8='RESULT'): RawUTF8;
var P, Name, Value: PUTF8Char;
    EndOfObject: AnsiChar;
begin
  result := '';
  P := pointer(JSON);
  if P=nil then
    exit;
  while P^<>'{' do
    if P^=#0 then
      exit else
      inc(P);
  inc(P); // jump {
  repeat
    Name := GetJSONField(P,P,nil,@EndOfObject);
    if (P=nil) or (Name=nil) or (EndOfObject<>':') then
      exit; // invalid Name:Value separator
    Value := GetJSONField(P,P,nil,@EndOfObject);
    if (Value=nil) or not(EndOfObject in [',','}']) then
      exit; // invalid item separator
    if IdemPChar(Name,pointer(aUpperName)) then begin
      Result := RawUTF8(Value);
      exit;
    end;
  until (P=nil) or (EndOfObject='}');
end;

const
  EndOfJSONValueField: set of AnsiChar = [#0,',','}',']'];
  EndOfJSONField: set of AnsiChar = [',',']','}',':'];
  DigitChars: set of AnsiChar = ['-','+','0'..'9'];

/// decode a JSON field into an UTF-8 encoded buffer, stored inplace of JSON data
function GetJSONField(P: PUTF8Char; out PDest: PUTF8Char;
  wasString: PBoolean=nil; EndOfObject: PUTF8Char=nil): PUTF8Char;
// this code is very fast
var U: PUTF8Char;
    w: word;
    c4: integer;
begin
  if wasString<>nil then
    wasString^ := false; // default is 'no string'
  PDest := nil; // mark error or unexpected end (#0)
  result := nil;
  if P=nil then exit;
  if P^<=' ' then repeat inc(P); if P^=#0 then exit; until P^>' ';
  c4 := PInteger(P)^ and $DFDFDFDF;
  if (c4=ord('N')+ord('U')shl 8+ord('L')shl 16+ord('L')shl 24) and
    (P[4] in EndOfJSONValueField)  then begin
    result := nil; // null value -> return nil
    inc(P,3);
  end else
  if (c4=ord('F')+ord('A')shl 8+ord('L')shl 16+ord('S')shl 24) and
    (ord(P[4]) and $DF=ord('E')) and (P[5] in EndOfJSONValueField) then begin
    PWord(P)^ := ord('0');
    result := P; // false -> return '0'
    inc(P,4);
  end else
  if (c4=ord('T')+ord('R')shl 8+ord('U')shl 16+ord('E')shl 24) and
    (P[4] in EndOfJSONValueField)  then begin
    PWord(P)^ := ord('1');
    result := P; // true -> return '1'
    inc(P,3);
  end else
  if P^='"' then begin
    // '"string \"\\field"' -> 'string "\field'
    if wasString<>nil then
      wasString^ := true;
    inc(P);
    U := P;
    result := P;
    repeat // unescape P^ into U^ (cf. http://www.ietf.org/rfc/rfc4627.txt)
      case P^ of
      #0:  exit;  // leave PDest=nil for unexpected end
      '"': break; // end of string
      '\': begin
        inc(P);
        case P^ of // unescape JSON string
          'b': U^ := #08;
          't': U^ := #09;
          'n': U^ := #$0a;
          'f': U^ := #$0c;
          'r': U^ := #$0d;
          'u': 
          if HexToBin(pointer(P+1),@w,2) then begin // '\u0123'
            w := swap(w);  // reverse byte order
            if w<=$7f then // speed up UTF-8 encoding for most used chars
              U^ := ansichar(w) else
              inc(U,UnicodeCharToUtf8(U,w)-1);
            inc(P,4);
          end else
            U^ := '?'; // bad formated hexa number -> '?0123'
          else U^ := P^; // litterals: '\"' -> '"'
        end;
      end;
      else U^ := P^;
      end; // case P^ of
      inc(U);
      inc(P);
    until false;
    U^ := #0; // make zero-terminated
  end else begin
    // numerical field: all chars before end of field
    if not (P^ in DigitChars) then // is first char (at least) a number?
      exit; // leave PDest=nil for unexpected end
    result := P;
  end;
  repeat // go to end of field
    inc(P);
    if P^=#0 then
      exit; // leave PDest=nil for unexpected end
  until P^ in EndOfJSONField;
  if EndOfObject<>nil then
    EndOfObject^ := P^;
  P^ := #0; // make zero-terminated
  PDest := @P[1];
  if P[1]=#0 then
    PDest := nil;
end;

function IsJSONString(P: PUTF8Char): boolean;  // test if P^ is a "string" value
var c4: integer;
begin
  if P^=' ' then
    repeat inc(P) until P^<>' ';
  c4 := PInteger(P)^ and $DFDFDFDF;
  if ((c4=ord('N')+ord('U')shl 8+ord('L')shl 16+ord('L')shl 24) and
        (P[4] in EndOfJSONValueField)) or
     ((c4=ord('F')+ord('A')shl 8+ord('L')shl 16+ord('S')shl 24) and
        (ord(P[4]) and $DF=ord('E')) and (P[5] in EndOfJSONValueField)) or
     ((c4=ord('T')+ord('R')shl 8+ord('U')shl 16+ord('E')shl 24) and
        (P[4] in EndOfJSONValueField)) then begin
    result := false; // constants are no string
    exit;
  end else
    if P^ in DigitChars then begin // is first char numeric?
      // check if P^ is a true numerical value
      repeat inc(P) until not (P^ in ['0'..'9']); // check digits
      if P^='.' then
        repeat inc(P) until not (P^ in ['0'..'9']); // check fractional digits
      if byte(P^) and $DF=ord('E') then begin
        inc(P);
        if P^='+' then inc(P) else
        if P^='-' then inc(P);
        while P^ in ['0'..'9'] do inc(P);
      end;
      if P^=' ' then
        repeat inc(P) until P^<>' ';
      result := (P^<>#0);
      exit;
    end else
      result := true; // don't begin with a numerical value -> must be a string
end;


{ ************ Unit-Testing classes and functions }

function KB(bytes: cardinal): RawUTF8;
begin
  if bytes>=1024*1024 then begin
    if bytes>=1024*1024*1024 then begin
      bytes := bytes div (1024*1024);
      result := ' GB';
    end else
      result := ' MB';
    result :=
      {$ifndef ENHANCEDRTL}Int32ToUtf8{$else}IntToStr{$endif}(bytes div (1024*1024))+'.'+
      {$ifndef ENHANCEDRTL}Int32ToUtf8{$else}IntToStr{$endif}((bytes mod (1024*1024))div (102*1024))+
      result;
  end else
  if bytes>1023*9 then
   result := {$ifndef ENHANCEDRTL}Int32ToUtf8{$else}IntToStr{$endif}(bytes div 1024)+' KB' else
   result := {$ifndef ENHANCEDRTL}Int32ToUtf8{$else}IntToStr{$endif}(bytes)+' B';
end;

function MicroSecToString(Micro: Int64): RawUTF8;
function TwoDigitToString(value: cardinal): RawUTF8;
var L: integer;
begin
  result := {$ifndef ENHANCEDRTL}Int32ToUtf8{$else}IntToStr{$endif}(value);
  L := length(result);
  if L=1 then
    result := '0.0'+result else // '3' -> '0.03'
  if L=2 then
    result := '0.'+result else // '35' -> '0.35'
    insert('.',result,L-1); // '103' -> '1.03'
end;
begin
  if Micro<1000 then
    result := {$ifndef ENHANCEDRTL}Int64ToUtf8{$else}IntToStr{$endif}(Micro)+'us' else
  if Micro<1000*1000 then
    result := TwoDigitToString(Micro div 10)+'ms' else
    result := TwoDigitToString(Micro div (10*1000))+'s';
end;

{$ifdef MSWINDOWS}

{ TPrecisionTimer }

function TPrecisionTimer.ByCount(Count: cardinal): RawUTF8;
begin
  if Count=0 then
    result := '0' else // avoid div per 0 exception
    result := MicroSecToString(iTime div Count);
end;

function TPrecisionTimer.PerSec(Count: cardinal): cardinal;
begin
  if iTime=0 then
    result := 0 else // avoid div per 0 exception
    result := (Int64(Count)*(1000*1000))div iTime;
end;

procedure TPrecisionTimer.Start;
begin
  QueryPerformanceCounter(iStart);
end;

function TPrecisionTimer.Stop: RawUTF8;
var Freq: Int64;
begin
  QueryPerformanceCounter(iStop);
  QueryPerformanceFrequency(Freq);
  if Freq=0 then
    iTime := 0 else
    iTime := ((iStop-iStart)*(1000*1000))div Freq;
  result := MicroSecToString(iTime);
end;

function TPrecisionTimer.Time: RawUTF8;
begin
  result := MicroSecToString(iTime);
end;

{$endif MSWINDOWS}

function GetDelphiCompilerVersion: RawUTF8;
begin
result := {$if defined(VER90)} 'Delphi 2'
{$elseif defined(VER100)} 'Delphi 3'
{$elseif defined(VER120)} 'Delphi 4'
{$elseif defined(VER130)} 'Delphi 5'
{$elseif defined(VER140)} 'Delphi 6'
{$elseif defined(VER150)} 'Delphi 7'
{$elseif defined(VER160)} 'Delphi 8'
{$elseif defined(VER170)} 'Delphi 2005'
{$elseif defined(VER185)} 'Delphi 2007'
{$elseif defined(VER180)} 'Delphi 2006'
{$elseif defined(VER200)} 'Delphi 2009'
{$elseif defined(VER210)} 'Delphi 2010'
{$elseif defined(FPC)}    'Free Pascal'
{$ifend}
end;

{ TSynTest }

procedure TSynTest.Add(aMethod: TSynTestEvent; const aName: string);
var i: integer;
begin
  if self=nil then
    exit; // avoid GPF
  i := Length(fTests);
  SetLength(fTests,i+1);
  fTests[i].TestName := aName;
  fTests[i].Method := aMethod;
end;

constructor TSynTest.Create(const Ident: string);
procedure AddChildrenFirst(C: TClass);
type
  TMethodInfo = packed record
    Len: Word;
    Addr: Pointer;
    Name: ShortString;
  end;
var Table: PWordArray;
    M: ^TMethodInfo;
    Method: TMethod;
    i: integer;
    text: RawUTF8;
begin
  if C=nil then
    exit;
  AddChildrenFirst(C.ClassParent); // add children published methods first
  Table := PPointer(PtrInt(C)+vmtMethodTable)^;
  if Table=nil then
    exit;
  Method.Data := self;
  M := @Table^[1];
  for i := 1 to Table^[0] do begin // Table^[0] = methods count
    inc(fInternalTestsCount);
    Method.Code := M^.Addr;
    text := M.Name;
    if M.Name[1]='_' then
      delete(text,1,1) else
      text := UnCamelCase(text);
    Add(TSynTestEvent(Method),string(text));
    inc(PtrInt(M),M^.Len);
  end;
end;
begin
  fIdent := Ident;
  AddChildrenFirst(PPointer(Self)^); // use recursion for adding
end;

function TSynTest.GetCount: Integer;
begin
  if self=nil then
    result := 0 else
    result := length(fTests);
end;

function TSynTest.GetIdent: string;
var tmp: RawUTF8;
begin
  if self=nil then
    result := '' else
  if fIdent<>'' then
    result := fIdent else begin
    tmp := RawUTF8(ClassName);
    if IdemPChar(Pointer(tmp),'TSYN') then
      if IdemPChar(Pointer(tmp),'TSYNTEST') then
        Delete(tmp,1,8) else
      Delete(tmp,1,4) else
    if IdemPChar(Pointer(tmp),'TTEST') then
      Delete(tmp,1,5) else
    if tmp[1]='T' then
      Delete(tmp,1,1);
    Result := string(UnCamelCase(tmp));
  end;
end;

function TSynTest.GetTestMethod(Index: integer): TSynTestEvent;
begin
  if Cardinal(Index)>=Cardinal(length(fTests)) then
    result := nil else
    result := fTests[Index].Method;
end;

function TSynTest.GetTestName(Index: integer): string;
begin
  if Cardinal(Index)>=Cardinal(length(fTests)) then
    result := '' else
    result := fTests[Index].TestName;
end;


{ TSynTestCase }

function TSynTestCase.Check(condition: Boolean; const msg: string): Boolean;
begin
  if not condition then
    TestFailed(msg);
  Inc(fAssertions);
  Result := not condition;
end;

function TSynTestCase.CheckNot(condition: Boolean; const msg: string): Boolean;
begin
  result := Check(not condition, msg);
end;

constructor TSynTestCase.Create(Owner: TSynTests; const Ident: string);
begin
  inherited Create(Ident);
  fOwner := Owner;
end;

function TSynTestCase.RandomString(CharCount: Integer): RawByteString;
var V: cardinal;
    P: PAnsiChar;
begin
  Result := '';
  SetLength(Result,CharCount);
  P := pointer(Result);
  while CharCount>0 do begin
    if CharCount>5 then begin
      V := Random(maxInt); // fast: one random compute per 5 chars
      P[0] := AnsiChar(32+V and 127); V := V shr 7;
      P[1] := AnsiChar(32+V and 127); V := V shr 7;
      P[2] := AnsiChar(32+V and 127); V := V shr 7;
      P[3] := AnsiChar(32+V and 127); V := V shr 7;
      P[4] := AnsiChar(65+V);
      Inc(P,5);
      dec(CharCount,5);
    end else begin
      P^ := AnsiChar(32+Random(224));
      inc(P);
      dec(CharCount);
    end;
  end;
end;

function TSynTestCase.RandomUTF8(CharCount: Integer): RawUTF8;
begin
  result := WinAnsiToUtf8(WinAnsiString(RandomString(CharCount)));
end;

procedure TSynTestCase.TestFailed(const msg: string);
begin
  if Owner<>nil then // avoid GPF
    Owner.fFailed.AddObject(msg,self);
  Inc(fAssertionsFailed);
end;


{ TSynTests }

procedure TSynTests.AddCase(TestCase: TSynTestCase);
begin
  TestCase.fMethodIndex := fCurrentMethod;
  TestCase.fTestCaseIndex := fTestCase.Count-fCurrentMethodFirstTestCaseIndex;
  fTestCase.Add(TestCase);
end;

procedure TSynTests.AddCase(const TestCase: array of TSynTestCaseClass);
var i: integer;
begin
  for i := low(TestCase) to high(TestCase) do
    AddCase(TestCase[i].Create(self));
end;

constructor TSynTests.Create(const Ident: string);
begin
  inherited;
  fFailed := TStringList.Create;
  fTestCase := TObjectList.Create;
end;

destructor TSynTests.Destroy;
begin
  fFailed.Free;
  fTestCase.Free; // TObjectList will free all TSynTestCase instance
  if TTextRec(fSaveToFile).Handle<>0 then
    Close(fSaveToFile);
end;

procedure TSynTests.DuringRun(TestCaseIndex, TestMethodIndex: integer);
var C: TSynTestCase;
    Run,Failed: integer;
begin
  // only handle console by default
  if ToConsole=nil then
    Exit;
  C := TestCase[TestCaseIndex];
  if C=nil then
    Exit;
  if TestMethodIndex<0 then
    writeln(ToConsole^,#13#10' ',C.MethodIndex+1,'.',C.TestCaseIndex+1,
      '. ',C.Ident,': ') else begin
    Run := C.Assertions-C.fAssertionsBeforeRun;
    Failed := C.AssertionsFailed-C.fAssertionsFailedBeforeRun;
    if Failed=0 then begin
      Write(ToConsole^,'  - ',C.TestName[TestMethodIndex],': ');
      if Run=0 then
        Writeln(ToConsole^,'no assertion') else
      if Run=1 then
        Writeln(ToConsole^,'1 assertion passed') else
        Writeln(ToConsole^,Run,' assertions passed');
    end else
      Writeln(ToConsole^,'!  - ',C.TestName[TestMethodIndex],': ',
        Failed,' / ',Run,' FAILED'); // ! to highlight the line
    if C.fRunConsole<>'' then begin
      Writeln(ToConsole^,'     ',C.fRunConsole);
      C.fRunConsole := '';
    end;
    if TestMethodIndex=C.Count-1 then begin
      Write(ToConsole^,'  Total failed: ',C.AssertionsFailed,' / ',C.Assertions,
        '  - ',C.Ident);
      if C.AssertionsFailed=0 then
        Writeln(ToConsole^,' PASSED') else
        Writeln(ToConsole^,' FAILED');
    end;
  end;
end;

function TSynTests.GetFailedCase(Index: integer): TSynTestCase;
begin
  if (self=nil) or (cardinal(Index)>=cardinal(fFailed.Count)) then
    Result := nil else begin
    Index := integer(fFailed.Objects[index]);
    if Index<InternalTestsCount then
      Result := nil else
      Result := TSynTestCase(Index);
  end;
end;

function TSynTests.GetFailedCaseIdent(Index: integer): string;
begin
  if (self=nil) or (cardinal(Index)>=cardinal(fFailed.Count)) then
    Result := '' else begin
    Index := integer(fFailed.Objects[index]);
    if Index<InternalTestsCount then
      // if integer(Objects[]) is lower than InternalTestsCount, then
      // it is an index to the corresponding published method, and the Strings[]
      // contains the associated failure message
      Result := TestName[Index] else
      // if integer(Objects[]) is equal or higher than InternalTestsCount,
      // the Objects[] points to the TSynTestCase, and the Strings[] to the
      // associated failure message
      Result := TSynTestCase(Index).Ident;
  end;
end;

function TSynTests.GetFailedCount: integer;
begin
  if self=nil then
    result := 0 else
    result := fFailed.Count;
end;

function TSynTests.GetFailedMessage(Index: integer): string;
begin
  if (self=nil) or (cardinal(Index)>=cardinal(fFailed.Count)) then
    result := '' else
    result := fFailed.Strings[Index];
end;

function TSynTests.GetTestCase(Index: integer): TSynTestCase;
begin
  if (self=nil) or (cardinal(Index)>=cardinal(fTestCase.Count)) then
    Result := nil else
    Result := TSynTestCase(fTestCase[Index]);
end;

function TSynTests.GetTestCaseCount: Integer;
begin
  if self=nil then
    result := 0 else
    result := fTestCase.Count;
end;

function TSynTests.Run: Boolean;
var i,t,m: integer;
    Elapsed, Version: RawUTF8;
    C: TSynTestCase;
{$ifdef MSWINDOWS}
    Vers: TFileVersion;
{$endif}
begin
  if ToConsole<>nil then
    Writeln(ToConsole^,#13#10'   ',Ident,#13#10'  ',StringOfChar('-',length(Ident)+2));
{$ifdef MSWINDOWS}
  RunTimer.Start;
  Vers.RetrieveApplicationVersion(paramstr(0),0);
{$endif}
  C := nil;
  try
    // 1. register all test cases
    fTestCase.Clear;
    for m := 0 to Count-1 do begin
      C := pointer(m);
      fCurrentMethod := m;
      fCurrentMethodFirstTestCaseIndex := fTestCase.Count;
      // published methods will call AddCase() to register tests in fTestCase[]
      TSynTestEvent(TestMethod[m]);
    end;
    // 2. launch the tests
    Randomize;
    fFailed.Clear;
    fAssertions := 0;
    fAssertionsFailed := 0;
    m := -1;
    for i := 0 to fTestCase.Count-1 do begin
        C := TestCase[i];
        if (ToConsole<>nil) and (C.MethodIndex<>m) then begin
          m := C.MethodIndex;
          writeln(ToConsole^,#13#10#13#10,m+1,'. ',TestName[m]);
        end;
        DuringRun(i,-1);
        C.fAssertions := 0; // reset assertions count
        C.fAssertionsFailed := 0;
        for t := 0 to C.Count-1 do begin
          C.fAssertionsBeforeRun := C.fAssertions;
          C.fAssertionsFailedBeforeRun := C.fAssertionsFailed;
          TSynTestEvent(C.TestMethod[t]); // run tests + Check() and TestFailed()
          DuringRun(i,t);
        end;
        inc(fAssertions,C.fAssertions); // compute global assertions count
        inc(fAssertionsFailed,C.fAssertionsFailed);
      end;
  except
    on E: Exception do
      // assume an exception not intercepted above is a failure
      fFailed.AddObject(E.ClassName+': '+E.Message,C);
  end;
  result := (fFailed.Count=0);
{$ifdef MSWINDOWS}
  Elapsed := #13#10#13#10'Time elapsed for all tests: '+RunTimer.Stop;
  if Vers.Major<>0 then
    Version := #13#10'Software version tested: '+RawUTF8(Vers.Detailed);
{$endif}
  if ToConsole<>nil then begin
    Writeln(ToConsole^,#13#10,Version,CustomVersions,
      #13#10'Generated with: ',GetDelphiCompilerVersion,' compiler',
      Elapsed,
      #13#10'Tests performed at ',DateTimeToStr(Now),
      #13#10#13#10'Total assertions failed for all test suits:  ',
      AssertionsFailed,' / ',Assertions);
    if result then
      Writeln(ToConsole^,#13#10'! All tests passed successfully.') else begin
      Writeln(ToConsole^,#13#10'! Some tests FAILED: please correct the code.');
{      for i := 0 to FailedCount do
        if FailedMessage[i]<>'' then
          Writeln(ToConsole^,'- ',FailedCaseIdent[i],' : ',FailedMessage[i]); } 
    end;
  end;
end;

procedure TSynTests.SaveToFile(const DestPath, FileName: TFileName);
var FN: TFileName;
begin
  {$I-}
  ToConsole := nil;
  if TTextRec(fSaveToFile).Handle<>0 then
    Close(fSaveToFile);
  if FileName='' then
    FN := DestPath+Ident+'.txt' else
    FN := DestPath+FileName;
  assign(fSaveToFile,FN);
  rewrite(fSaveToFile);
  if IOResult=0 then
    ToConsole := @fSaveToFile else
    fillchar(fSaveToFile,sizeof(fSaveToFile),0);
  {$I+}
end;


{ TTestLowLevelCommon }

procedure TTestLowLevelCommon._CamelCase;
var v: RawUTF8;
begin
  v := UnCamelCase('On'); Check(v='On');
  v := UnCamelCase('ON'); Check(v='ON');
  v := UnCamelCase('OnLine'); Check(v='On line');
  v := UnCamelCase('OnLINE'); Check(v='On LINE');
  v := UnCamelCase('OnMyLINE'); Check(v='On my LINE');
end;

procedure TTestLowLevelCommon.Bits;
var Bits: array[byte] of byte;
    Bits64: Int64 absolute Bits;
    Si,i: integer;
begin
  fillchar(Bits,sizeof(Bits),0);
  for i := 0 to high(Bits)*8+7 do
    Check(not GetBit(Bits,i));
  RandSeed := 10; // will reproduce the same Random() values
  for i := 1 to 100 do begin
    Si := Random(high(Bits));
    SetBit(Bits,Si);
    Check(GetBit(Bits,Si));
  end;
  RandSeed := 10;
  for i := 1 to 100 do
    Check(GetBit(Bits,Random(high(Bits))));
  RandSeed := 10;
  for i := 1 to 100 do begin
    Si := Random(high(Bits));
    UnSetBit(Bits,Si);
    Check(not GetBit(Bits,Si));
  end;
  for i := 0 to high(Bits)*8+7 do
    Check(not GetBit(Bits,i));
  for i := 0 to 63 do
    Check(not GetBit64(Bits,i));
  RandSeed := 10;
  for i := 1 to 30 do begin
    Si := Random(63);
    SetBit64(Bits64,Si);
    Check(GetBit64(Bits,Si));
  end;
  RandSeed := 10;
  for i := 1 to 30 do
    Check(GetBit64(Bits,Random(63)));
  RandSeed := 10;
  for i := 1 to 30 do begin
    Si := Random(63);
    UnSetBit64(Bits64,Si);
    Check(not GetBit64(Bits,Si));
  end;
  for i := 0 to 63 do
    Check(not GetBit64(Bits,i));
  Randomize; // we fixed the RandSeed value above -> get true random now
end;

procedure TTestLowLevelCommon.Curr64;
var tmp: string[63];
    i, err: Integer;
    V1: currency;
    V2: extended;
    i64: Int64;
    v: RawUTF8;
begin
  Check(StrToCurr64(pointer(Curr64ToStr(1)))=1);
  Check(StrToCurr64(pointer(Curr64ToStr(12)))=12);
  Check(StrToCurr64(pointer(Curr64ToStr(123)))=123);
  Check(StrToCurr64(pointer(Curr64ToStr(1234)))=1234);
  Check(StrToCurr64(pointer(Curr64ToStr(12340000)))=12340000);
  Check(StrToCurr64(pointer(Curr64ToStr(12345000)))=12345000);
  Check(StrToCurr64(pointer(Curr64ToStr(12345600)))=12345600);
  Check(StrToCurr64(pointer(Curr64ToStr(12345670)))=12345670);
  tmp[0] := AnsiChar(Curr64ToPChar(1,@tmp[1])); Check(tmp='0.0001');
  tmp[0] := AnsiChar(Curr64ToPChar(12,@tmp[1])); Check(tmp='0.0012');
  tmp[0] := AnsiChar(Curr64ToPChar(123,@tmp[1])); Check(tmp='0.0123');
  tmp[0] := AnsiChar(Curr64ToPChar(1234,@tmp[1])); Check(tmp='0.1234');
  for i := 1 to 5000 do begin
    V1 := Random*1000000;
    v := Curr64ToStr(PInt64(@V1)^);
    tmp[0] := AnsiChar(Curr64ToPChar(PInt64(@V1)^,@tmp[1]));
    Check(RawUTF8(tmp)=v);
    V2 := GetExtended(pointer(v),err);
    Check(err=0);
    Check(Abs(V1-V2)<1E-10);
    i64 := StrToCurr64(pointer(v));
    Check(PInt64(@V1)^=i64);
  end;
end;

procedure TTestLowLevelCommon.FastStringCompare;
begin
  Check(StrIComp('abcD','ABcd')=0);
  Check(StrIComp('abcD','ABcF')=StrComp('ABCD','ABCF'));
  Check(StrComp('ABCD','ABCD')=0);
  Check(AnsiIComp('abcD','ABcd')=0);
  Check(AnsiIComp('abcD','ABcF')=StrComp('ABCD','ABCF'));
  Check(StrIComp('abcD','ABcd')=AnsiIComp('abcD','ABcd'));
  Check(StrIComp('abcD','ABcF')=AnsiIComp('ABCD','ABCF'));
end;

procedure TTestLowLevelCommon.IniFiles;
var Content,S,N,V: RawUTF8;
    Si,Ni,Vi,i,j: integer;
procedure Debug;
begin
  with TFileStream.Create('testini.ini',fmCreate) do begin
    Write(pointer(Content)^,length(Content));
    Free;
  end;
end;
begin
  Content := '';
  Randomize;
  //RandSeed := 10;
  for i := 1 to 1000 do begin
    Si := Random(20);
    Ni := Random(50);
    Vi := Si*Ni+Ni;
    if Si=0 then
      S := '' else
      S := 'Section'+{$ifndef ENHANCEDRTL}Int32ToUtf8{$else}IntToStr{$endif}(Si);
    N := {$ifndef ENHANCEDRTL}Int32ToUtf8{$else}IntToStr{$endif}(Ni);
    V := {$ifndef ENHANCEDRTL}Int32ToUtf8{$else}IntToStr{$endif}(Vi);
    UpdateIniEntry(Content,S,N,V);
    for j := 1 to 5 do
      Check(FindIniEntry(Content,S,N)=V,'FindIniEntry');
    Check(FindIniEntry(Content,S,'no')='');
    Check(FindIniEntry(Content,'no',N)='');
  end;
//  Debug; write('Enter');readln;
end;

procedure TTestLowLevelCommon.Soundex;
var e: cardinal;
    PC: PAnsiChar;
    Soundex: TSynSoundEx;
begin
  Check(SoundExAnsi(' 120 ')=0);
  if SOUNDEX_BITS=8 then
    e := $2050206 else
    e := $2526;
  Check(SoundExAnsi('bonjour')=e);
  Check(SoundExAnsi(' 123 bonjour.  m',@PC)=e);
  Check((PC<>nil) and (PC^='.'));
  Check(SoundExAnsi(' 123 bonjourtrslongmotquidpasse  m',@PC)<>0);
  Check((PC<>nil) and (PC^=' '));
  Check(SoundExAnsi('BOnjour')=e);
  Check(SoundExAnsi('Bnjr')=e);
  Check(SoundExAnsi('bonchour')=e);
  Check(SoundExAnsi('mohammad')=SoundExAnsi('mohhhammeeet'));
  if SOUNDEX_BITS=8 then
    e := $2050206 else
    e := $25262;
  Check(SoundExAnsi('bonjours')=e);
  Check(SoundExAnsi('BOnjours')=e);
  Check(SoundExAnsi('Bnjrs')=e);
  Check(SoundExAnsi(' 120 ')=0);
  if SOUNDEX_BITS=8 then
    e := $2050206 else
    e := $2526;
  Check(SoundExUTF8('bonjour')=e);
  Check(SoundExUTF8(' 123 bonjour.  m',@PC)=e);
  Check((PC<>nil) and (PC^=' '));
  Check(SoundExUTF8(' 123 bonjourtrslongmotquidpasse  m',@PC)<>0);
  Check((PC<>nil) and (PC^=' '));
  Check(SoundExUTF8('BOnjour')=e);
  Check(SoundExUTF8('Bnjr')=e);
  Check(SoundExUTF8('bonchour')=e);
  Check(SoundExUTF8('mohammad')=SoundExUTF8('mohhhammeeet'));
  if SOUNDEX_BITS=8 then
    e := $2050206 else
    e := $25262;
  Check(SoundExUTF8('bonjours')=e);
  Check(SoundExUTF8('BOnjours')=e);
  Check(SoundExUTF8('Bnjrs')=e);
  Check(Soundex.Prepare('mohamad'));
  Check(Soundex.Ansi('moi rechercher mohammed ici'));
  Check(Soundex.UTF8('moi rechercher mohammed ici'));
end;

procedure TTestLowLevelCommon.SystemCopyRecord;
type TR = record
       One: integer;
       S1: AnsiString;
       Three: byte;
       S2: WideString;
       Five: boolean;
{$ifndef LVCL}
       V: Variant; {$endif}
       R: Int64Rec;
       Arr: array[0..10] of AnsiString;
       Dyn: array of integer;
       Bulk: array[0..9] of byte;
     end;
var A,B,C: TR;
begin
  if pos('Synopse framework',Owner.CustomVersions)=0 then
    Owner.CustomVersions := Owner.CustomVersions+#13#10'Synopse framework used: '+
      SYNOPSE_FRAMEWORK_VERSION;
  fillchar(A,sizeof(A),0);
  A.S1 := 'one';
  A.S2 := 'two';
  A.Five := true;
  A.Three := $33;
{$ifndef LVCL}
  A.V := 'One Two';
{$endif}
  A.R.Lo := 10;
  A.R.Hi := 20;
  A.Arr[5] := 'five';
  SetLength(A.Dyn,10);
  A.Dyn[9] := 9;
  B := A;
  Check(A.One=B.One);
  Check(A.S1=B.S1);
  Check(A.Three=B.Three);
  Check(A.S2=B.S2);
  Check(A.Five=B.Five);
  {$ifndef LVCL} Check(A.V=B.V); {$endif}
  Check(Int64(A.R)=Int64(B.R));
  Check(A.Arr[5]=B.Arr[5]);
  Check(A.Arr[0]=B.Arr[0]);
  Check(A.Dyn[9]=B.Dyn[9]);
  Check(A.Dyn[0]=0);
  B.Three := 3;
  B.Dyn[0] := 10;
  C := B;
  Check(A.One=C.One);
  Check(A.S1=C.S1);
  Check(C.Three=3);
  Check(A.S2=C.S2);
  Check(A.Five=C.Five);
  {$ifndef LVCL} Check(A.V=C.V); {$endif}
  Check(Int64(A.R)=Int64(C.R));
  Check(A.Arr[5]=C.Arr[5]);
  Check(A.Arr[0]=C.Arr[0]);
  Check(A.Dyn[9]=C.Dyn[9]);
  {Check(A.Dyn[0]=0) bug in original VCL?}
  Check(C.Dyn[0]=10);
end;

procedure TTestLowLevelCommon.UrlEncoding;
var i: integer;
    s: RawByteString;
begin
  Check(UrlEncode('abcdef')='abcdef');
  Check(UrlEncode('abcdefyzABCDYZ01239_-.~ ')='abcdefyzABCDYZ01239_-.~+');
  Check(UrlEncode('"Aardvarks lurk, OK?"')='%22Aardvarks+lurk%2C+OK%3F%22');
  for i := 0 to 100 do begin
    s := RandomString(i*5);
    Check(UrlDecode(UrlEncode(s))=s,string(s));
  end;
end;

procedure TTestLowLevelCommon.NumericalConversions;
var i, j, err: integer;
    k,l: Int64;
    s: RawUTF8;
    d,e: double;
    a: shortstring;
    u: string;
begin
  for i := 1 to 10000 do begin
    j := Random(maxInt)-Random(maxInt);
    str(j,a);
    s := RawUTF8(a);
    u := string(a);
    Check(IntToStr(j)=u);
    Check(Int32ToUtf8(j)=s);
    Check(format('%d',[j])=u);
    Check(GetInteger(pointer(s))=j);
    Check(FormatUTF8('%',[j])=s);
    k := Int64(j)*Random(MaxInt);
    str(k,a);
    s := RawUTF8(a);
    u := string(a);
    Check(IntToStr(k)=u);
    Check(Int64ToUtf8(k)=s);
    Check(format('%d',[k])=u);
    Check(FormatUTF8('%',[k])=s);
    err := 1;
    l := GetInt64(pointer(s),err);
    Check((err=0)and(l=k));
    str(j,a);
    Check(IntToStr(j)=string(a));
    Check(format('%d',[j])=string(a));
    Check(format('%.8x',[j])=IntToHex(j,8));
    d := Random*1E-17-Random*1E-9;
    str(d,a);
    s := RawUTF8(a);
    e := GetExtended(Pointer(s),j);
    Check(SameValue(e,d)); // test str()
    s := ExtendedToStr(d,15);
    e := GetExtended(Pointer(s),j);
    Check(SameValue(e,d)); 
  end;
end;

procedure TTestLowLevelCommon._UTF8;
var i: integer;
    W: WinAnsiString;
    U: RawUTF8;
    Unic: RawUnicode;
    WA: Boolean;
begin
  for i := 0 to 1000 do begin
    W := WinAnsiString(RandomString(i*5));
    Check(length(W)=i*5);
    U := WinAnsiToUtf8(W);
    Check(Utf8ToWinAnsi(U)=W);
    Unic := Utf8DecodeToRawUnicode(U);
    Check(RawUnicodeToUtf8(Unic)=U);
    Check(RawUnicodeToWinAnsi(Unic)=W);
    WA := IsWinAnsi(pointer(Unic));
    Check(IsWinAnsi(pointer(Unic),length(Unic)shr 1)=WA);
    Check(IsWinAnsiU(pointer(U))=WA);
  end;
end;

initialization
  InitSynCommonsConversionTables;
end.
